<?php

namespace App\Services\Reports;

use App\Models\AssetStatus;
use App\Models\AssetType;
use App\Models\Country;
use App\Models\Location;
use App\Models\MakeAndModel;
use App\Models\Manufacturer;
use App\Models\Region;
use App\Models\SiteCode;
use App\Models\TechnicalSpecs;
use App\Repositories\Reports\AvailableStockRepository;
use App\Repositories\CommonFilterRepository;
use App\Http\Responses\ReportOutputData;

class AvailableStockService extends AbstractReportService
{

    protected $repository;
    protected $commonFilterRepository;
    protected $reportOutputData;

    /**
     * Constructs a new instance of the class.
     *
     * @param AvailableStockRepository $repository The AvailableStockRepository instance.
     * @param CommonFilterRepository $commonFilterRepository The repository for the CommonFilter entity.
     * @param ReportOutputData $reportOutputData The ReportOutputData entity
     */
    public function __construct(AvailableStockRepository $repository, CommonFilterRepository $commonFilterRepository, ReportOutputData $reportOutputData)
    {
        $this->repository = $repository;
        $this->commonFilterRepository = $commonFilterRepository;
        $this->reportOutputData = $reportOutputData;
    }

    /**
     * Retrieves the filter data options for various fields.
     *
     * @return array The filter data options for asset types, hardware standards, technical specifications, statuses, manufacturers, locations, countries, regions, and site codes.
     */
    public function getFilterDataOptions()
    {

        $filterDataOptions['assetTypes'] = AssetType::select('id', 'name')->valid()->orderBy('name', 'ASC')->get();;
        $filterDataOptions['hardwareStds'] = MakeAndModel::select('id', 'name')->valid()->orderBy('name', 'ASC')->get();;
        $filterDataOptions['techSpecs'] = TechnicalSpecs::whereNotNull('details')->valid()->orderBy('details', 'ASC')->get();;
        $filterDataOptions['statuses'] = AssetStatus::select('id','name')->whereIn('slug', config('reports.available-stock.include_statuses'))->orderBy('name','ASC')->get();
        $filterDataOptions['manufacturers'] = Manufacturer::select('id','name')->orderBy('name','ASC')->get();
        $filterDataOptions['locations'] = Location::warehouses()->select('id','room_name')->orderBy('room_name')->get();
        $filterDataOptions['countries'] = Country::select('id','name')->orderBy('name')->get();
        $filterDataOptions['regions'] = Region::select('id','name')->orderBy('name')->get();
        $filterDataOptions['siteCodes'] = SiteCode::has('locations')->select('id','code')->orderBy('code')->get();

        return $filterDataOptions;
    }

    /**
     * Making query for the report data.
     *
     * @return array
     */
    public function data()
    {
        $stocks = $this->repository->getAvailableStockDetails();
        $stocks = $this->filterWithInputData($stocks);
        $stockResult = $stocks->get();
        $count = $stockResult->count();
        $totalAssetCount = $stockResult->sum('asset_count');
        $stocks = $this->reportOutputData->getOutputData($stocks, ['model' => 'asc']);

        return compact('stocks', 'count', 'totalAssetCount');
    }

    /**
     * Making the query for input filters.
     *
     * @param object $stocks
     * @return object
     */
    public function filterWithInputData($stocks)
    {
        $inputData = request()->form ?? request()->all();
        $locationStatus = $inputData['location_status'] == 'active' ? 1 : (($inputData['location_status'] == 'inactive') ? '0' : '');

        $stocks = $this->commonFilterRepository->filterWithDirectFields($stocks, 'locations.id', $inputData['location_id'] ?? '');
        $stocks = $this->commonFilterRepository->filterWithDirectFields($stocks, 'locations.region_id', $inputData['region_id'] ?? '');
        $stocks = $this->commonFilterRepository->filterWithDirectFields($stocks, 'locations.country_id', $inputData['country_id'] ?? '');
        $stocks = $this->commonFilterRepository->filterWithDirectFields($stocks, 'assets.asset_type_id', $inputData['asset_type_id'] ?? '');
        $stocks = $this->commonFilterRepository->filterWithDirectFields($stocks, 'assets.asset_status_id', $inputData['asset_status_id'] ?? '');
        $stocks = $this->commonFilterRepository->filterWithDirectFields($stocks, 'assets.make_and_model_id', $inputData['make_and_model_id'] ?? '');
        $stocks = $this->repository->filterTechRelation($stocks, 'assets.technical_spec_id', $inputData['technical_spec_id'] ?? '', $inputData['make_and_model_id'] ?? '');
        $stocks = $this->commonFilterRepository->filterWithDirectFields($stocks, 'manufacturers.id', $inputData['manufacturer_id']  ?? '');
        $stocks = $this->commonFilterRepository->filterWithDirectFields($stocks, 'site_codes.id', $inputData['site_code_id']  ?? '');
        $stocks = $this->commonFilterRepository->filterWithDirectFields($stocks, 'locations.status', $locationStatus);

        return $stocks;
    }

    /**
     * Creating array for table data.
     *
     * @return array
     */
    public function getNestedData($item, $index)
    {
        $nestedData = [];
        $commonData = $this->getReportNestedData($item);
        $nestedData = $commonData['report'];
        $nestedData['id']   = $index;
        $nestedData['location'] = generateLocationLink($item->location_id, $item->location);

        return $nestedData;
    }

    /**
     * Making the query for export.
     *
     * @return object
     */
    public function exportData()
    {
        $stocks = $this->repository->getAvailableStockDetails();
        $stocks = $this->filterWithInputData($stocks);
        $stocks = $stocks->orderBy('model', 'asc');

        return $stocks;
    }

    /**
     * Creating array for export data.
     *
     * @param $item Query Result
     *
     * @return array
     */
    public function getReportNestedData($item)
    {
        $nestedData['report']['asset_type'] = $nestedData['export']['Asset Type'] = disableCSVInjection($item->type ?? '');
        $nestedData['report']['manufacturer'] = $nestedData['export']['Manufacturer'] = disableCSVInjection($item->manufacturer ?? '');
        $nestedData['report']['hardware_standard'] = $nestedData['export']['Hardware Standard'] = disableCSVInjection($item->model ?? '');
        $nestedData['report']['tech_specs'] = $nestedData['export']['Technical Specs'] = disableCSVInjection($item->tech_specs ?? '');
        $nestedData['report']['asset_status'] = $nestedData['export']['Asset Status'] = disableCSVInjection($item->status ?? '');
        $nestedData['report']['site_code'] = $nestedData['export']['Site Code'] = disableCSVInjection($item->site_code ?? '');
        $nestedData['report']['location'] = $nestedData['export']['Location'] = disableCSVInjection($item->location ?? '');
        $nestedData['report']['country'] = $nestedData['export']['Country'] = disableCSVInjection($item->country ?? '');
        $nestedData['report']['region'] = $nestedData['export']['Region'] = disableCSVInjection($item->region ?? '');
        $nestedData['report']['asset_count'] = $nestedData['export']['# of Assets'] = $item->asset_count ?? '';

        return $nestedData;
    }
}
