<?php

namespace App\Services\Reports;

use App\Http\Responses\ReportOutputData;
use App\Repositories\CommonFilterRepository;
use App\Repositories\Reports\AssetsUsedRepository;

class AssetsReusedService extends AbstractReportService
{
    /**
     * Constructor method for the class.
     *
     * @param AssetsUsedRepository $repository The repository for assets used.
     * @param CommonFilterRepository $commonFilterRepository The repository for common filters.
     * @param ReportOutputData $reportOutputData The report output data.
     */
    public function __construct(protected AssetsUsedRepository $repository, protected CommonFilterRepository $commonFilterRepository, protected ReportOutputData $reportOutputData)
    {
        $this->repository = $repository;
        $this->commonFilterRepository = $commonFilterRepository;
        $this->reportOutputData = $reportOutputData;
    }

    /**
     * Retrieves the requested data from the form and processes it.
     *
     * @return array Returns an array containing the filtered reused assets and the count of assets.
     */
    public function data()
    {
        $inputData  = $this->getInputData();
        $result     = $this->repository->getReusedAssetsData();
        $result     = $this->filterWithInputData($result, $inputData);
        $count      = $result->count();
        $result     = $this->reportOutputData->getOutputData($result, ['id' => 'desc']);

        return compact('result', 'count');
    }

    /**
     * Filters the given query with the provided input data.
     *
     * @param mixed $query The query to be filtered.
     * @param array $inputData The input data used for filtering.
     * @return mixed The filtered query.
     */
    public function filterWithInputData($query, $inputData)
    {
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'asset_tag', $inputData['asset_tag']);
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'serial_no', $inputData['serial_no']);
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'asset_type_id', $inputData['asset_type_id']);
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'make_and_model_id', $inputData['make_and_model_id']);
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'technical_spec_id', $inputData['technical_spec_id']);
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'asset_status_id', $inputData['asset_status_id']);
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'user_id', $inputData['user_id']);
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'location_id', $inputData['location_id']);
        $query = $this->repository->filterWithReusedCount($query, $inputData['no_of_times']);
        $query = $this->repository->filterAssetsByStatus($query);

        return $query;
    }

    /**
     * Retrieves the input data for processing.
     *
     * @return array The input data with default values for missing keys.
     */
    public function getInputData()
    {
        $requestedData = request()->form ?? request()->all();

        return [
            'asset_tag' => $requestedData['asset_tag'] ?? '',
            'serial_no' => $requestedData['serial_no'] ?? '',
            'asset_type_id' => $requestedData['asset_type'] ?? '',
            'make_and_model_id' => $requestedData['make_and_model'] ?? '',
            'technical_spec_id' => $requestedData['technical_spec'] ?? '',
            'asset_status_id' => $requestedData['status'] ?? '',
            'user_id' => $requestedData['user'] ?? '',
            'location_id' => $requestedData['location'] ?? '',
            'no_of_times' => $requestedData['no_of_times'] ?? ''
        ];
    }

    /**
     * Create output data
     * @params $assets,$start, $data, $type
     * @return array
     * */
    public function getOutputData($assets, $data)
    {
        $parentIndex = 0;

        foreach ($assets as $key => $asset) {
            ++$parentIndex;
            $nestedData = $this->getNestedData($asset, $parentIndex);
            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Retrieves the nested data for a given item and index.
     *
     * @param mixed $item The item for which to retrieve the nested data.
     * @param int $index The index of the item.
     * @return array The nested data for the item.
     */
    public function getNestedData($item, $index)
    {
        $commonData = $this->getReportNestedData($item);
        $nestedData = $commonData['report'];
        $nestedData['id'] = $index;
        $nestedData['serial_no'] = generateAssetLink($item->id, $item->serial_no);
        $nestedData['asset_tag'] = generateAssetLink($item->id, $item->asset_tag);
        $nestedData['user'] = generateUserLink($item->user?->id, $item->user?->user_name);
        $nestedData['asset_location'] = generateLocationLink($item->location?->id, $item->location?->room_name);

        return $nestedData;
    }

    /**
     * Retrieves the nested data for generating a report.
     *
     * @param $asset The asset object for which the report is being generated.
     */
    public function getReportNestedData($asset)
    {
        $user_details = optional($asset->user)->first_name . ' ' . optional($asset->user)->last_name;

        if ($asset->user) {
            $user_details .= ', ' . optional($asset->user)->location;
        }

        $nestedData = [];
        $nestedData['report']['asset_tag']          = $nestedData['export']['Asset Tag #'] = $asset->asset_tag;
        $nestedData['report']['serial_no']          = $nestedData['export']['Serial #'] = $asset->serial_no;
        $nestedData['report']['asset_type']         = $nestedData['export']['Asset Type'] = $asset?->assetType?->name;
        $nestedData['report']['make_and_model']     = $nestedData['export']['Hardware Standard'] = $asset->makeAndModel?->makeModelName;
        $nestedData['report']['technical_spec']     = $nestedData['export']['Technical Specs'] = $asset->technicalSpec?->details;
        $nestedData['report']['asset_status']       = $nestedData['export']['Asset Status'] = $asset?->assetStatus?->name;
        $nestedData['report']['asset_location']     = $nestedData['export']['Location'] = $asset->location?->room_name;
        $nestedData['report']['user']               = $nestedData['export']['User'] = $user_details;
        $nestedData['report']['age_of_device']      = $nestedData['export']['Asset Age'] = $asset->AssetAge;
        $nestedData['report']['no_of_time_used']    = $nestedData['export']['# of Times Reused'] = $asset->reused_asset_history_count;
        $nestedData['report']['latest_assigned_at'] = $nestedData['export']['Last Assigned'] = $asset->latestAssignedAssetHistory?->created_date;

        return $nestedData;
    }

    /**
     * Exports the data.
     *
     * @return mixed
     */
    public function exportData()
    {
        $inputData = $this->getInputData();
        $reusedAssets = $this->repository->getReusedAssetsData();
        $assets = $this->filterWithInputData($reusedAssets, $inputData);
        $assets = $assets->orderBy('id', 'desc');

        return $assets;
    }


    /**
     * Get count of reused assets
     * @return count
     * */
    public function getReUsedAssetCount()
    {
        $reusedAssetData = $this->repository->getReusedAssetsData()->get()->count();

        return $reusedAssetData;
    }

    /**
     * Get count of not reused assets
     * @return count
     * */
    public function getNotReUsedAssetCount()
    {
        $notReusedAssetData = $this->repository->getNotReusedAssetsData()->get()->count();

        return $notReusedAssetData;
    }
}
