<?php

namespace App\Services\Reports;

use App\Http\Responses\ReportOutputData;
use App\Models\AssetType;
use App\Repositories\CommonFilterRepository;
use App\Repositories\Reports\AssetsCreatedRepository;
use Carbon\Carbon;

class AssetsCreatedService extends AbstractReportService
{
    protected $repository;
    protected $commonFilterRepository;
    protected $reportOutputData;

    /**
     * Constructor for initializing the AssetsCreatedRepository, CommonFilterRepository, and ReportOutputData.
     *
     * @param AssetsCreatedRepository $repository repository for assets created
     * @param CommonFilterRepository $commonFilterRepository repository for common filter
     * @param ReportOutputData $reportOutputData report output data
     */
    public function __construct(AssetsCreatedRepository $repository, CommonFilterRepository $commonFilterRepository, ReportOutputData $reportOutputData)
    {
        $this->repository = $repository;
        $this->commonFilterRepository = $commonFilterRepository;
        $this->reportOutputData = $reportOutputData;
    }

    /**
     * Sets the asset type filter data based on the selected asset type.
     *
     * @param array $assetTypesAll The array containing all asset types.
     * @return mixed The filtered asset types based on the selected asset type.
     */
    public function setAssetTypeFilterData()
    {
        $inputData = $this->getInputData();
        $selectedAssetType = $inputData['asset_types'] ?? [];

        if (!empty($selectedAssetType)) {
            if (is_array($selectedAssetType)) {
                return AssetType::whereIn('id', $selectedAssetType)->get();
            }
            
            return AssetType::where('id', $selectedAssetType)->get();
        }

        return [];
    }

    /**
     * getReportData
     *
     * @return object
     */
    public function data()
    {
        $inputData = $this->getInputData();

        if (empty($inputData['asset_types'])) {
            $result = null;
            $count  = 0;

            return compact('result', 'count');
        }
        $assetCreatedDataResponse = $this->getAssetCreatedData($inputData);
        $result = $this->filterWithInputData($assetCreatedDataResponse, $inputData);
        $count  = $result->count();
        $result = $this->reportOutputData->getOutputData($result, ['id' => 'desc']);

        return compact('result', 'count');
    }

    /**
     * Retrieves input data based on the provided request data.
     *
     * @return array
     */
    public function getInputData(): array
    {
        $requestedData = request()->form ?? request()->all();

        return [
            'asset_types' => $requestedData['asset_types'] ?? '',
            'date_from' => $requestedData['date_from'] ?? '',
            'date_to' => $requestedData['date_to'] ?? '',
            'date' => $requestedData['date'] ?? '',
            'selectedAssetTypeBar' => $requestedData['selectedAssetTypeBar'] ?? '',
        ];
    }

    /**
     * Retrieves the nested data for a given item.
     *
     * @param mixed $item The item for which to retrieve the nested data.
     * @param int $index The index of the item.
     * @return array The nested data for the item.
     */
    public function getNestedData($item, $index)
    {
        $commonData = $this->getReportNestedData($item);
        $nestedData = $commonData['report'];
        $nestedData['id'] = $index;
        $nestedData['asset_tag'] = generateAssetLink($item->id, $item->asset_tag);
        $nestedData['serial_no'] = generateAssetLink($item->id, $item->serial_no);
        $nestedData['user_location'] = $item->user ? generateUserLink($item->user?->id, $nestedData['user_location']) : generateLocationLink($item->location?->id, $nestedData['user_location']);

        return $nestedData;
    }

    /**
     * Filter assets based on input data.
     *
     * @param datatype $assets description
     * @param array $inputData description
     * @return assets filtered based on input data
     */
    public function filterWithInputData($assets, $inputData)
    {
        if (!empty($inputData['selectedAssetTypeBar'])) {
            $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'asset_type_id', $inputData['selectedAssetTypeBar']);
        } else {
            $assets = $this->commonFilterRepository->filterWithDirectFields($assets, 'asset_type_id', $inputData['asset_types']);
        }

        return $assets;
    }

    /**
     * Get nested data for the report.
     *
     * @param datatype $asset description of the asset parameter
     * @return array the nested data for the report
     */
    public function getReportNestedData($asset)
    {
        $nestedData = [];
        $nestedData['report']['asset_tag']              = $nestedData['export']['Asset Tag #']            = disableCSVInjection($asset->asset_tag);
        $nestedData['report']['serial_no']              = $nestedData['export']['Serial #']               = disableCSVInjection($asset->serial_no);
        $nestedData['report']['asset_type']             = $nestedData['export']['Asset Type']             = disableCSVInjection($asset->assetType ? $asset->assetType->name : '');
        $nestedData['report']['hardware_standard']      = $nestedData['export']['Hardware Standard']      = disableCSVInjection($asset->makeAndModel ? $asset->makeAndModel->makeModelName : '');
        $nestedData['report']['technical_specs']        = $nestedData['export']['Technical Specs']        = disableCSVInjection($asset->technicalSpec ? $asset->technicalSpec->details : '');
        $nestedData['report']['user_location']          = $nestedData['export']['User/Location']          = disableCSVInjection($asset->user ? ($asset->user->userName) : ($asset->location ? $asset->location->room_name : ''));
        $nestedData['report']['created_at']             = $nestedData['export']['Created Date']           = disableCSVInjection($asset->created_at);

        return $nestedData;
    }

    /**
     * getReportData
     *
     * @return object
     */
    public function exportData()
    {
        $inputData = $this->getInputData();
        $assetCreatedDataResponse = $this->getAssetCreatedData($inputData);
        $assets = $this->filterWithInputData($assetCreatedDataResponse, $inputData);
        $assets = $assets->orderBy('id', 'desc');

        return $assets;
    }

    /**
     * Retrieves the date from the request or defaults to the first day of the current month from last year.
     *
     * @return Carbon The date retrieved from the request or the default date.
     */
    public function getFromDate($inputData)
    {
        //Date coming form bar
        $date = $inputData['date'] ?? '';

        if (!empty($date)) {
            $dateFrom = Carbon::createFromFormat("M Y", $date)->startOfMonth();

            return $dateFrom;
        } 
        
        
        $dateFrom = $inputData['date_from'] ?? '';
        
        if (!empty($dateFrom)) {
            $dateFrom = format_date_to_carbon($dateFrom);
        } else {
            $dateFrom = first_day_current_month_last_year();
        }

        return $dateFrom;
    }

    /**
     * Retrieves the "to date" from the given request or defaults to the current day
     *
     * @return \Carbon\Carbon The "to date" value.
     */
    public function getToDate($inputData)
    {

        //Date  coming form bar
        $date = $inputData['date'] ?? '';

        if (!empty($date)) {
            $dateFrom = Carbon::createFromFormat("M Y", $date)->endOfMonth();

            return $dateFrom;
        } 

       $dateTo = $inputData['date_to'] ?? '';
        
        if (!empty($dateTo)) {
            $dateTo = format_date_to_carbon($dateTo);
        } else {
            $dateTo = Carbon::now();
        }

        return $dateTo;
    }

    /**
     * Get formatted data based on assets and months.
     *
     * @param mixed $assets The assets data
     * @param array $months The months data
     * @return array The formatted data
     */
    public function getFormattedData($assets, $months)
    {
        $formattedData = [];

        foreach ($months as $monthNumber => $monthName) {
            $formattedData[] = is_countable($assets->get($monthNumber)) ? count($assets->get($monthNumber)) : 0;
        }

        return $formattedData;
    }

    /**
     * Get data for the graph representation
     *
     * @return array
     */
    public function getGraphData()
    {
        $inputData = $this->getInputData();
        $assetCreatedDataResponse = $this->getAssetCreatedData($inputData);
        $assetCreatedDataResponse = $this->repository->getAssetCreatedCountSelectQuery($assetCreatedDataResponse);
        $assetCreatedDataResponse = $assetCreatedDataResponse->get();
        
        $assetTypes = $this->setAssetTypeFilterData();
        $dateFrom = $this->getFromDate($inputData);
        $dateTo = $this->getToDate($inputData);
        $months = get_past_month_names($dateFrom, $dateTo);
        $data = [];

        foreach($assetTypes as $assetType) {
            $asset = $assetCreatedDataResponse->where('asset_type_id', $assetType->id);
			$assetsGroupedByMonth =  group_by_months($asset);
            $data[$assetType->slug] = $this->getFormattedData($assetsGroupedByMonth, $months);
		}
        
        $monthNames = collect($months)->flatten()->toArray();

        return ['data' => $data, 'months' => $monthNames, 'assetTypes' => $assetTypes];

    }

    /**
     * common function for retrive asset created data based on input data.
     *
     * @param array $inputData
     * @return mixed
     */
    public function getAssetCreatedData($inputData)
    {
        $dateFrom = $this->getFromDate($inputData);
        $dateTo = $this->getToDate($inputData);
        $assetCreatedQuery = $this->repository->getAssetCreatedData($dateFrom, $dateTo);

        return $assetCreatedQuery;
    }

}
