<?php

namespace App\Services\Reports;

use App\Repositories\Reports\AssetsByDepartmentRepository;
use App\Http\Responses\ReportOutputData;
use App\Http\Traits\DepreciationValueTrait;

class AssetsByDepartmentService extends AbstractReportService
{

    use DepreciationValueTrait;

    protected $repository;
    protected $reportOutputData;

    /**
     * __construct
     *
     * @param  AssetsByDepartmentRepository $repository
     * @param  ReportOutputData $reportOutputData
     * @return void
     */
    public function __construct(AssetsByDepartmentRepository $repository, ReportOutputData $reportOutputData)
    {
        $this->repository = $repository;
        $this->reportOutputData = $reportOutputData;
    }

    /**
     * Retrieve the chart data for the department assets.
     *
     * @return mixed The chart data for the department assets.
     */
    public function getChartData()
    {
        return $this->repository->departmentAssetChart();
    }

    /**
     * Retrieves all departments.
     * @return departments data
     */
    public function getAllDepartments()
    {
        return $this->repository->getAllDepartments();
    }

    /**
     * Retrieves the data for the given function.
     *
     * @return array contains the report data
     */
    public function data()
    {
        $requestedData = request()->form ?? request()->all();

        if (empty($requestedData['departments'])) {
            $count = 0;
            $assets = [];
            $additionalDataForReport = [];
            $additionalDataForReport['totalDepreciatedValue'] = config('currency.currency_symbol') . 0;
            $additionalDataForReport['totalOriginalValue'] = config('currency.currency_symbol') . 0;

            return compact('count', 'assets', 'additionalDataForReport');
        }
        $departmentId = $requestedData['departments'] ?? 0;
        $assets = $this->repository->getDepartmentData($departmentId);
        $count = $assets->count();
        $additionalDataForReport = [];

        $assets = $this->reportOutputData->getOutputData($assets, ['created_at' => 'desc']);

        // for widgets total values
        $additionalDataForReport['totalDepreciatedValue'] = config('currency.currency_symbol'). number_format($this->getDepreciationValue($this->repository->getDepartmentData($departmentId)));
        $additionalDataForReport['totalOriginalValue'] = config('currency.currency_symbol'). number_format($this->getOriginalValue($this->repository->getDepartmentData($departmentId)));

        return compact('count', 'assets', 'additionalDataForReport');
    }

    /**
     * Export the data.
     *
     * @return mixed
     */
    public function exportData()
    {
        $requestedData = request()->form ?? request()->all();
        $departmentId = $requestedData['departments'] ?? 0;
        $assets = $this->repository->getDepartmentData($departmentId);
        $assets = $assets->orderBy('created_at', 'desc');

        return $assets;
    }


    /**
     * Format the record for export as csv
     * @param object single record object
     *
     * @return array formatted
     */
    public function getReportNestedData($asset)
    {
        $nestedData['report']['asset_tag'] = $nestedData['export']['Asset Tag #'] = disableCSVInjection($asset->asset_tag ?? '');
        $nestedData['report']['serial_no'] = $nestedData['export']['Serial #'] = disableCSVInjection($asset->serial_no ?? '');
        $nestedData['report']['asset_type'] = $nestedData['export']['Asset Type'] = disableCSVInjection($asset->assetType?->name);
        $nestedData['report']['make_and_model'] = $nestedData['export']['Hardware Standard'] = disableCSVInjection(optional($asset->makeAndModel)->makeModelName);
        $nestedData['report']['tech_specs'] = $nestedData['export']['Technical Specs'] = disableCSVInjection(optional($asset->technicalSpec)->details);
        $nestedData['report']['assigned_to'] = $nestedData['export']['Assigned To'] = disableCSVInjection(optional($asset->user)->username);
        $orginalValue = $asset->technicalSpec ? (!empty($asset->technicalSpec->original_value) ? $asset->technicalSpec->original_value : 0) : 0;
        $nestedData['report']['original_value'] = $nestedData['export']['Original Value'] = config('currency.currency_symbol') . number_format($orginalValue);
        $depreciatedValue = !empty($asset->DepreciatedValue) ? $asset->DepreciatedValue : 0;
        $nestedData['report']['depreciated_value'] = $nestedData['export']['Depreciated Value'] = config('currency.currency_symbol') . number_format($depreciatedValue);


        return $nestedData;
    }

    /**
     * Get the nested data for a given asset and index.
     *
     * @param mixed $asset The asset object.
     * @param mixed $index The index of the asset.
     * @return array The nested data.
     */
    public function getNestedData($asset, $index)
    {
        $commonData = $this->getReportNestedData($asset);
        $nestedData = $commonData['report'];
        $userLabelWithLink  = $asset->user ? generateUserLink($asset->user->id, $nestedData['assigned_to']) : '';
        $nestedData['id']   = $index;
        $nestedData['asset_tag'] = generateAssetLink($asset->id, $nestedData['asset_tag']);
        $nestedData['serial_no'] = generateAssetLink($asset->id, $nestedData['serial_no']);
        $nestedData['assigned_to'] = $userLabelWithLink;

        return $nestedData;
    }
}
