<?php

namespace App\Services\Reports;

use App\Http\Responses\ReportOutputData;
use App\Repositories\Reports\AssetsUsedRepository;
use App\Services\Reports\AbstractReportService;
use App\Repositories\CommonFilterRepository;

class AssetsAssignedService extends AbstractReportService
{
    /**
     * Constructs a new instance of the class.
     *
     * @param AssetsUsedRepository $repository The repository for assets used.
     * @param CommonFilterRepository $commonFilterRepository The repository for common filters.
     * @param ReportOutputData $reportOutputData The data for report output.
     */
    public function __construct(protected AssetsUsedRepository $repository, protected CommonFilterRepository $commonFilterRepository, protected ReportOutputData $reportOutputData)
    {
        $this->repository = $repository;
        $this->commonFilterRepository = $commonFilterRepository;
        $this->reportOutputData = $reportOutputData;
    }

    /**
     * filter and return assets data and totalcount
     * @param
     * @return assetData & TotalCount
     */
    public function data()
    {
        $requestedData = request()->form;
        $noOfTimesAssigned = $requestedData['no_of_times'] ?? '';
        $assignedAssetData = $this->getAssignedAssets($noOfTimesAssigned);
        $inputData = $this->getInputData($requestedData);
        $inputData['asset_ids']  = $assignedAssetData['asset_id'];

        if ($noOfTimesAssigned && empty($inputData['asset_ids'])) {
            return array('result' => collect(), 'count' => 0);
        }
        $result = $this->repository->getAssetsData();
        $result = $this->filterWithInputData($result, $inputData);
        $count  = $result->count();
        $result = $this->reportOutputData->getOutputData($result, ['id' => 'desc']);
        $result = $this->repository->mapAssetsWithAssetCount($result, $assignedAssetData['count']);

        return compact('result', 'count');
    }

    /**
     * Filter the query results based on the filters applied
     *
     * @param  mixed $assets
     * @param  mixed $inputData
     * @return object
     */
    public function filterWithInputData($query, $inputData)
    {
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'serial_no', $inputData['serial_no']);
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'asset_type_id', $inputData['asset_type_id']);
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'asset_status_id', $inputData['asset_status_id']);
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'make_and_model_id', $inputData['make_and_model_id']);
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'technical_spec_id', $inputData['technical_spec_id']);
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'id', $inputData['asset_ids']);

        return $query;
    }

    /**
     * Retrieves the input data from the requested data array.
     *
     * @param array $requestedData The requested data array.
     * @return array The input data array with the following keys:
     */
    public function getInputData($requestedData)
    {
        return [
            'serial_no' => $requestedData['serial_no'] ?? '',
            'asset_type_id' => $requestedData['asset_type'] ?? '',
            'asset_status_id' => $requestedData['status'] ?? '',
            'make_and_model_id' => $requestedData['make_and_model'] ?? '',
            'technical_spec_id' => $requestedData['technical_spec'] ?? '',
        ];
    }

    /**
     * Retrieves the nested data for generating a report.
     *
     * @param $asset The asset object for which the report is being generated.
     */
    public function getReportNestedData($asset)
    {
        $nestedData = [];
        $nestedData['report']['asset_tag']              = $nestedData['export']['Asset Tag #'] = $asset->asset_tag;
        $nestedData['report']['serial_no']              = $nestedData['export']['Serial #'] = $asset->serial_no;
        $nestedData['report']['asset_type']             = $nestedData['export']['Asset Type'] = $asset?->assetType?->name;
        $nestedData['report']['hardware_standard']      = $nestedData['export']['Hardware Standard'] = $asset->makeAndModel?->makeModelName;
        $nestedData['report']['tech_specs']             = $nestedData['export']['Technical Specs'] = $asset->technicalSpec?->details;
        $nestedData['report']['asset_status']           = $nestedData['export']['Asset Status'] = $asset?->assetStatus?->name;
        $nestedData['report']['no_of_time_assigned']    = $nestedData['export']['# of Times Assigned'] = $asset->count ?? 0;

        return $nestedData;
    }

    /**
     * Create output data
     * @params $assets,$start, $data, $type
     * @return array
     * */
    public function getOutputData($assets, $data)
    {
        $parentIndex = 0;

        foreach ($assets as $asset) {
            ++$parentIndex;
            $nestedData = $this->getNestedData($asset, $parentIndex);
            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Create each row of output data
     * @params $assets,$index,$type
     * @return array
     * */
    public function getNestedData($item, $index)
    {
        $commonData = $this->getReportNestedData($item);
        $nestedData = $commonData['report'];
        $nestedData['id'] = $index;
        $nestedData['asset_tag'] = generateAssetLink($item->id, $item->asset_tag);
        $nestedData['serial_no'] = generateAssetLink($item->id, $item->serial_no);

        return $nestedData;
    }

    /**
     * Exports the data.
     *
     * @return mixed
     */
    public function exportData()
    {
        $requestedData = request()->all();
        $inputData = $this->getInputData($requestedData);
        $noOfTimesAssigned = $requestedData['no_of_times'] ?? '';
        $assignedAssetData = $this->getAssignedAssets($noOfTimesAssigned);
        $assets = $this->repository->getAssetsData();
        $inputData['asset_ids'] = $assignedAssetData['asset_id'];
        $assets = $this->filterWithInputData($assets, $inputData);
        $assets = $assets->orderBy('id', 'desc');

        return $assets;
    }

    public function mapAssetsWithAssetCount($assignedAssets)
    {
        $noOfTimesAssigned = request('no_of_times') ?? '';
        $assignedAssetData = $this->getAssignedAssets($noOfTimesAssigned);

        return $this->repository->mapAssetsWithAssetCount($assignedAssets, $assignedAssetData['count']);
    }

    /**
     * find assigned asset ids and each asset assigned count
     * @param noOfTimesAssigned (request)
     * @return array an combined array of assetIds and each assetIds assigned counts
     */
    public function getAssignedAssets($noOfTimesAssigned)
    {
        $assignedAssetsData = $this->repository->getAssignedAssetsFromAssetHistory($noOfTimesAssigned);
        $assignedAssetId = $this->filterArrayData($assignedAssetsData, 'asset_id');
        $assignedAssetCount = $this->filterArrayData($assignedAssetsData, 'count', 'asset_id');

        return ['asset_id' => $assignedAssetId, 'count' => $assignedAssetCount];
    }

    /**
     * function to perform array_column
     * @params $data, $columnKey, $indexKey
     * @return array
     * */
    public function filterArrayData($data, $columnKey, $indexKey = null)
    {
        return is_array($data) ? array_column($data, $columnKey, $indexKey) : [];
    }
}
