<?php

namespace App\Services\Reports;

use App\Repositories\Reports\AssetsInTheStatusByMonthRepository;
use App\Http\Responses\ReportOutputData;
use App\Services\ReportGraphAndDataExport;
use Carbon\Carbon;
use Illuminate\Support\Facades\Storage;
use Maatwebsite\Excel\Facades\Excel;

/**
 * AssetsAssignedByMonthService - service class for assets inDepot to assigned avg time period calculation
 */
class AssetsAssignedByMonthService extends AbstractReportService
{
    protected $repository;
    protected $reportOutputData;


    /**
     * __construct
     *
     * @param  AssetsInTheStatusByMonthRepository $repository
     * @param ReportOutputData $reportOutputData The ReportOutputData entity
     * @return void
     */
    public function __construct(AssetsInTheStatusByMonthRepository $repository, ReportOutputData $reportOutputData)
    {
        $this->repository = $repository;
        $this->reportOutputData = $reportOutputData;
    }

    /**
    * get count of assets in  'assigned' and some specifc status in each months of a year
    * @param Int $year
    * @return Array $resultData
    */
    public function assetsAssignedByMonthData($year)
    {
        $inputData = $this->getInputData();
        $assetsAssignedByMonthData = $this->getAssetsAssignedByMonth($inputData);
        $assetsAssignedByMonthData = $this->repository->getAssetsSelectQuery($assetsAssignedByMonthData);

        return $this->formatAssetsAssignedByMonthData($assetsAssignedByMonthData);
    }

    /**
     * set zero to non existing months data and sort monthwise and Format data  to an array of months for barchart reporesentation
     * @param Array $assetsAssignedByMonthData
     * @return Array $resultData
     */
    public function formatAssetsAssignedByMonthData($assetsAssignedByMonthData)
    {
        $months = [1=>"January",2=>"February",3=>"March",4=>"April",5=>"May",6=>"June",7=>"July",8=>"August",9=>"September",10=>"October",11=>"November",12=>"December"];
        $assignedCount = array_column($assetsAssignedByMonthData, 'assigned_count', 'month');
        $brandnewAndUsedCount = array_column($assetsAssignedByMonthData, 'brandnew_and_used_count', 'month');

        foreach ($months as $month => $monthName) {

            if (!array_key_exists($month, $assignedCount)) {
                $assignedCount[$month] = "0";
            }

            if (!array_key_exists($month, $brandnewAndUsedCount)) {
                $brandnewAndUsedCount[$month] = "0";
            }
        }
        ksort($assignedCount);
        ksort($brandnewAndUsedCount);

        return array(
            'months' =>array_values($months),
            'assigned_count' =>array_values($assignedCount),
            'brandnew_and_used_count' => array_values($brandnewAndUsedCount),
        );
    }

    /**
     * Get data for the report table.
     *
     * @return array
     */
    public function data()
    {
        $inputData = $this->getInputData();
        $result = $this->getAssetsAssignedByMonth($inputData);
        $count  = $result->count();
        $result = $this->reportOutputData->getOutputData($result, ['created_at' => 'desc']);

        return compact('result', 'count');
    }

     /**
     * Retrieves the input data from the requested data array.
     *
     * @return array The input data array containing the following keys:
     */
    public function getInputData()
    {
        $requestedData = request()->form ?? request()->all();
        $inputData =  [
            'year'    => $requestedData['year'] ?? '',
            'date'    => $requestedData['date'] ?? '',
        ];

        return $inputData;
    }

     /**
     * Get Asset data that changed to assigned, brand new and used in between the date range
     *
     * @param array $inputData
     * @return object $asset
     */
    public function getAssetsAssignedByMonth($inputData)
    {
        $dateFrom = $this->getFromDate($inputData);
        $dateTo = $this->getToDate($inputData);
        $assetHistoryQuery = $this->repository->getAssetsListInInStatusesInYear($dateFrom, $dateTo);

        return $assetHistoryQuery;
    }

    /**
     * Get the start of date range
     * @return Carbon
     */
    public function getFromDate($inputData)
    {

        $date = $inputData['date'] ?? '';
        $year = $inputData['year'] ?? Carbon::now()->year;

        if (!empty($date)) {
            $yearDate = $date . ' ' . $year;
            $dateFrom = Carbon::createFromFormat("M Y", $yearDate)->startOfMonth();

            return $dateFrom;
        }


        $dateFrom = format_date_from_year_to_carbon($year)->startOfYear();

        return $dateFrom;
    }

    /**
     * Get the end of date range
     * @return Carbon
     */
    public function getToDate($inputData)
    {

        $date = $inputData['date'] ?? '';
        $year = $inputData['year'] ?? Carbon::now()->year;

        if (!empty($date)) {
            $yearDate = $date . ' ' . $year;
            $dateFrom = Carbon::createFromFormat("M Y", $yearDate)->endOfMonth();

            return $dateFrom;
        }


        $dateTo = format_date_from_year_to_carbon($year)->endOfYear();

        return $dateTo;
    }

    /**
     * Returns an array of nested data for exporting.
     *
     * @param mixed $asset The asset object.
     * @return array The nested data array.
     */
    public function getReportNestedData($assetHistory)
    {
        $nestedData = [];
        $nestedData['report']['asset_tag']          = $nestedData['export']['Asset Tag #'] = $assetHistory->asset?->asset_tag;
        $nestedData['report']['serial_no']          = $nestedData['export']['Serial #'] = $assetHistory->asset?->serial_no;
        $nestedData['report']['asset_type']         = $nestedData['export']['Asset Type'] = $assetHistory->asset?->assetType?->name;
        $nestedData['report']['hardware_standard']       = $nestedData['export']['Hardware Standard'] = $assetHistory->asset?->makeAndModel?->makeModelName;
        $nestedData['report']['tech_specs']         = $nestedData['export']['Technical Specs'] = $assetHistory->asset?->technicalSpec?->details;
        $nestedData['report']['asset_status']               = $nestedData['export']['Asset Status'] = disableCSVInjection($assetHistory->newStatus?->name);
        $nestedData['report']['current_asset_status']              = $nestedData['export']['Current Asset Status'] = disableCSVInjection($assetHistory->asset?->assetStatus?->name);

        return $nestedData;
    }

    /**
     * Retrieves nested data for a given asset and index.
     *
     * @param mixed $item The asset object.
     * @param int $index The index of the nested data.
     *
     * @return array The nested data.
     */
    public function getNestedData($item, $index)
    {
        $commonData = $this->getReportNestedData($item);
        $nestedData = $commonData['report'];
        $nestedData['id'] = $index;
        $nestedData['serial_no'] = generateAssetLink($item->asset?->id, $nestedData['serial_no']);
        $nestedData['asset_tag'] = generateAssetLink($item->asset?->id, $nestedData['asset_tag']);

        return $nestedData;
    }

    /**
     * Retrieves the export data based on the input data.
     *
     * @return mixed The export data.
     */
    public function exportData()
    {
        $inputData = $this->getInputData();
        $assets = $this->getAssetsAssignedByMonth($inputData);
        $assets = $assets->orderBy('created_at', 'desc');

        return $assets;
    }

    /**
	 * Export the graph and data list to Excel.
	 *
	 * @param array $dataCollection The collection of data.
	 */
	public function exportGraphAndDataListToExcel(array $dataCollection)
	{
		$name = str_replace(' ', '_', request('name'));
		$name = str_replace('/', '_', $name);
		$sheetTitle = str_replace(' ', '_', request('sheettitle'));
		$sheetTitle = str_replace('/', '_', $sheetTitle);

		$headings = array_keys($dataCollection[0][0]);
		$image = storage_path('app/' . request('image'));
		$filename = request("filename") . time() . '.xlsx';
		$excel = Excel::download(new ReportGraphAndDataExport($dataCollection, $image, $headings, $sheetTitle), $filename, \Maatwebsite\Excel\Excel::XLSX);
		$img = explode('public/', request('image'));
		Storage::disk('public')->delete($img[1]);

		return $excel;
	}


}
