<?php

namespace App\Services\Reports;

use App\Http\Responses\ReportOutputData;
use App\Repositories\Reports\ActiveUsersRepository;
use App\Repositories\CommonFilterRepository;

/**
 * service class for for Active Users Without Assets Report
 */
class ActiveUsersService  extends AbstractReportService
{
    
    protected $repository;
    protected $commonFilterRepository;
    protected $reportOutputData;
    protected $userRepository;


    /**
     * Initializes a new instance of the class.
     *
     * @param ActiveUsersRepository $repository The repository for active users.
     * @param CommonFilterRepository $commonFilterRepository The repository for common filters.
     * @param ReportOutputData $reportOutputData The output data for the report.
     */
    public function __construct(ActiveUsersRepository $repository, CommonFilterRepository $commonFilterRepository, ReportOutputData $reportOutputData)
    {
        $this->repository = $repository;
        $this->commonFilterRepository = $commonFilterRepository;
        $this->reportOutputData = $reportOutputData;
    }

    /**
     * Retrieves and processes data of active users without assets, then returns a compact array.
     * 
     * @return array Contains 'result' with the filtered and sorted list of users, and 'count' with the total number of users.
     */
    public function data()
    {
        $inputData  = $this->getInputData();
        $result     = $this->repository->getActiveUsersWithNoAssets();
        $result     = $this->filterResults($result, $inputData);
        $count      = $result->count();

        $result     = $this->reportOutputData->getOutputData($result, ['first_name' => 'asc']);

        return compact('result', 'count');
    }

    /**
     * Retrieves input data from the request.
     *
     * @return array
     */
    public function getInputData()
    {
        $requestedData = request()->form ?? request()->all();
        
        return [
            'searchText'      => $requestedData['searchText'] ?? '',
            'department'      => $requestedData['department'] ?? '',
            'city'            => $requestedData['city'] ?? '',
            'start_date_from' => $requestedData['start_date_from'] ?? '',
            'start_date_to'   => $requestedData['start_date_to'] ?? '',
        ];
    }

    /**
     * Filters the results based on the input data.
     *
     * @param array $users The array of assets to filter.
     * @param array $inputData The input data used for filtering.
     * @return object The filtered array of assets.
     */
    public function filterResults($users, $inputData)
    {
        $users = $this->repository->searchUserWithGeneralQueries($users, $inputData['searchText']);
        $users = $this->commonFilterRepository->filterWithDirectFields($users, 'department_id', $inputData['department']);
        $users = $this->commonFilterRepository->filterWithDirectFields($users, 'city', $inputData['city']);
        $users = $this->commonFilterRepository->filterWithDirectDateRange($users, 'hire_date', $inputData['start_date_from'], $inputData['start_date_to']);

        return $users;
    }

    /**
     * Export the data.
     *
     * @return mixed
     */
    public function exportData()
    {
        $inputData = $this->getInputData();
        $users = $this->repository->getActiveUsersWithNoAssets();
        $users = $this->filterResults($users, $inputData);
        $users = $users->orderBy('created_at', 'desc');

        return $users;
    }

     /**
     * Format the record for export as csv
     * @param object single record object
     * 
     * @return array formatted
     */
    public function getReportNestedData($user)
    {
        
        if (!empty($user->hire_date && $user->hire_date != '0000-00-00' && $user->hire_date != '11/30/-0001')) {
            $hireDate = $user->hire_date;
        } else {
            $hireDate = '';
        }
        
        $nestedData['report']['user_name'] = $nestedData['export']['User Name'] = disableCSVInjection($user->first_name . ' ' . $user->last_name);
        $nestedData['report']['email'] = $nestedData['export']['Email'] = disableCSVInjection($user->email);
        $nestedData['report']['location'] = $nestedData['export']['Location'] = disableCSVInjection($user->city);
        $nestedData['report']['manager_name'] = $nestedData['export']['Manager Name'] = disableCSVInjection(optional($user->manager)->first_name . ' ' . optional($user->manager)->last_name);
        $nestedData['report']['manager_email'] = $nestedData['export']['Manager Email'] = disableCSVInjection(optional($user->manager)->email);
        $nestedData['report']['department'] = $nestedData['export']['Department'] = disableCSVInjection(optional($user->department)->name);
        $nestedData['report']['start_date'] = $nestedData['export']['Start Date'] = disableCSVInjection($hireDate);
        $nestedData['report']['status'] = $nestedData['export']['Status'] = disableCSVInjection($user->status == 1 ? 'Active' : 'Terminated');
        
        return $nestedData;
    }

    /**
     * Generating data table data from qurey result
     * @param $users Query Result
     * @param $index  index
     * @return array
     */
    public function getNestedData($user, $index)
    {
        $commonData = $this->getReportNestedData($user);
        $nestedData = $commonData['report'];
        $userLink =  route('users.show', $user->id);
        $nestedData['id'] = $index;
        $nestedData['email'] = "<a href=$userLink>". $nestedData['email'] . "</a>";
        
        return $nestedData;
    }
}
