<?php

namespace App\Services\OperatingSystemCompliance;

use App\Models\OperatingSystemComplianceSettings;

class WindowsComplianceService extends OSComplianceService
{
    protected $settings;

    public function __construct()
    {
        parent::__construct();
        $this->settings = OperatingSystemComplianceSettings::select('os_platform', 'os_version')->where('device_type', 'windows')->first();
        $repositoryName = "\App\Repositories\OsCompliance\\" . config('os-compliance.windows_os.repository');
        $this->repository = new $repositoryName;
    }

    /**
     * Making query for the report
     */
    public function data()
    {
        $inputData  = $this->getInputData();
        $result     = $this->repository->getNonOsCompliantDevicesFromIntune($this->settings);
        $result     = $this->filterWindowsDevices($result, $inputData);
        $count      = $result->count();
        $result     = $this->getOutputData($result);
        return compact('result', 'count');
    }

    /**
     * Making the query for input filters
     */
    public function filterWindowsDevices($assets, $inputData)
    {
        foreach ($inputData as $key => $item) {
            if ($item != null) {
                if (in_array($key, ['mdm_serial_no', 'model', 'os_version', 'platform'])) {
                    if ($key == 'mdm_serial_no') {
                        $key = 'serial_no';
                    }
                    $assets = $this->intuneRepository->filterRelation($assets, $key, $item);
                }
                if ($key == 'user') {
                    $assets = $this->intuneRepository->filterRelation($assets, 'user_name', $item);
                }
                if (in_array($key, ['serial_no', 'asset_type_id', 'make_and_model_id', 'user_id', 'location_id', 'asset_status_id', 'technical_spec_id'])) {
                    $assets = $this->intuneRepository->filterWithAssetFields($assets, $key, $item);
                }
                if ($key == 'checkinFrom') {
                    $assets = $this->intuneRepository->filterCheckInDateFrom($assets, $item);
                }
                if ($key == 'checkinTo') {
                    $assets = $this->intuneRepository->filterCheckInDateTo($assets, $item);
                }
            }
        }
        return $assets;
    }

    public function getNestedData($item, $index)
    {
        $commonData = $this->getExportNestedData($item);
        $nestedData = $commonData['report'];
        $nestedData['id']   = $index;
        $nestedData['serial_no'] = $item->asset ? generateAssetLink($item->asset->id, $nestedData['serial_no']) : '';
        $nestedData['user_location'] =  $item->asset ? ($item->asset->user ? generateUserLink($item->asset->user->id, $item->asset->user->userName) : ($item->asset->location ? generateLocationLink($item->asset->location->id, $item->asset->location->room_name) : '')) : '';
        $nestedData['intune_user'] = $item->user_id ? generateUserLink($item->user_id, $item->email) : '';

        return $nestedData;
    }

    /**
     * Making the query for export
     */
    public function exportData()
    {
        $inputData = $this->getInputData();
        $assets = $this->repository->getNonOsCompliantDevicesFromIntune($this->settings);
        $assets = $this->filterWindowsDevices($assets, $inputData);
        $assets = $assets->orderBy('id', 'desc');

        return $assets;
    }

    /**
     * Creating array for export data
     * @param $item Query Result
     * @return array
     */
    public function getExportNestedData($item)
    {
        $nestedData['report']['serial_no'] = $nestedData['export']['Serial #'] = disableCSVInjection($item->asset ? $item->asset->serial_no : '');
        $nestedData['report']['asset_serial_no'] = $nestedData['export']['Serial # (' . config('os-compliance.windows_os.mdm') . ')'] = disableCSVInjection($item->serial_no ?? '');
        $nestedData['report']['asset_type'] = $nestedData['export']['Asset Type'] = disableCSVInjection($item->asset ? ($item->asset->assetType ? $item->asset->assetType->name : '') : '');
        $nestedData['report']['hardware_standard'] = $nestedData['export']['Hardware Standard (' . config('os-compliance.windows_os.mdm') . ')'] = disableCSVInjection($item->manufacturer ? $item->manufacturer . ' ' . $item->model : $item->model);
        $nestedData['report']['hardware_teqtivity'] = $nestedData['export']['Hardware Standard (Teqtivity)']   = disableCSVInjection(optional(optional($item->asset)->makeAndModel)->name);
        $nestedData['report']['technical_spec'] = $nestedData['export']['Technical Specs'] = disableCSVInjection($item->asset ? optional($item->asset->technicalSpec)->details : '');
        $nestedData['report']['asset_status'] = $nestedData['export']['Asset Status'] = disableCSVInjection($item->asset ? ($item->asset->assetStatus ? $item->asset->assetStatus->name : '') : '');
        $nestedData['report']['user_location'] = $nestedData['export']['User/Location'] = disableCSVInjection($item->asset ? ($item->asset->user ? $item->asset->user->userName : ($item->asset->location ? $item->asset->location->room_name : '')) : ($item->user ? $item->user->userName : ''));
        $nestedData['report']['intune_user'] = $nestedData['export']['User (' . config('os-compliance.windows_os.mdm') . ')'] =  disableCSVInjection($item->user_name ?? '');
        $nestedData['report']['os_platform'] = $nestedData['export']['OS Platform (' . config('os-compliance.windows_os.mdm') . ')'] = disableCSVInjection($item->platform ?? '');
        $nestedData['report']['os_version'] = $nestedData['export']['OS Version (' . config('os-compliance.windows_os.mdm') . ')'] = disableCSVInjection($item->os_version ?? '');
        $nestedData['report']['latest_version'] = $nestedData['export']['Latest Version'] = disableCSVInjection($this->settings ? $this->settings->os_version : '');
        $nestedData['report']['check_in_date'] = $nestedData['export']['Last Check In'] = $item->checkin_date ?? '';

        return $nestedData;
    }
}
