<?php

namespace App\Services\OperatingSystemCompliance;

use App\Models\AirWatch;
use App\Models\Chromebook;
use App\Models\IntuneComputer;
use App\Models\OperatingSystemComplianceSettings;
use App\Repositories\DiscoveryTools\WorkspaceOneRepository;
use App\Repositories\DiscoveryTools\ChromebookRepository;
use App\Repositories\DiscoveryTools\IntuneRepository;

/**
 * Service class for Operating System Compliance
 */
class OSComplianceService
{
    protected $repository;
    protected $workspaceOneRepository;
    protected $chromebookRepository;
    protected $intuneRepository;

    public function __construct()
    {
        $this->workspaceOneRepository = new WorkspaceOneRepository;
        $this->chromebookRepository = new ChromebookRepository;
        $this->intuneRepository = new IntuneRepository;
    }

    /**
     *  Get operating system compliance settings data
     */
    public function getSettingsFilters()
    {
        $result = [];
        $result['apple'] = $this->getOsData('apple');
        $result['windows'] = $this->getOsData('windows');
        $result['chrome'] = $this->getOsData('chrome');
        $result['android'] = $this->getOsData('android');
        $result['apple_mobile'] = $this->getOsData('apple_mobile');

        return $result;
    }

    /**
     *  fetching operating system compliance platform & version
     *  @param $type string device type
     *  @return array
     */
    public function getOsData($type)
    {
        $data = [];
        $settings = OperatingSystemComplianceSettings::select('os_platform', 'os_version')->where('device_type', $type)->first();
        if ($settings) {
            $data['platform'] = optional($settings)->os_platform ?? '';
            $data['version'] = optional($settings)->os_version ?? '';
        }

        return $data;
    }

    /**
     *  Get input from operating system compliance settings
     */
    public function getInputDataFromSettings($request)
    {
        $inputData = $data = array();
        foreach ($request as $key => $value) {
            if ($key == '_token') {
                continue;
            }
            $data['device_type'] = $key;
            $data['os_version'] = $value['os_version'] ?? null;
            $data['os_platform'] = $value['os_platform'] ?? null;
            $inputData[] = $data;
        }

        return $inputData;
    }

    /**
     *  Get input from operating system compliance report
     *  @return Array
     */
    public function getInputData()
    {
        $requestedData = request()->form ?? request()->all();

        return [
            'serial_no' => $requestedData['serial_no'] ?? '',
            'mdm_serial_no' => $requestedData['mdm_serial_no'] ?? '',
            'asset_type_id' => $requestedData['asset_type'] ?? '',
            'make_and_model_id' => $requestedData['make_and_model'] ?? '',
            'model' => $requestedData['model'] ?? '',
            'user_id' => $requestedData['user_id'] ?? '',
            'user' => $requestedData['user'] ?? '',
            'location_id' => $requestedData['location'] ?? '',
            'checkinFrom' => $requestedData['check_in_from'] ?? '',
            'checkinTo' => $requestedData['check_in_to'] ?? '',
            'asset_status_id' => $requestedData['asset_status_id'] ?? '',
            'technical_spec_id' => $requestedData['technical_spec'] ?? '',
            'os_version' => $requestedData['os_version'] ?? '',
            'platform' => $requestedData['os_platform'] ?? '',
        ];
    }

    /**
     * Making the query for asset filters
     */
    public function filterAssets($assets, $inputData)
    {
        $assets     = $assets->relationData($inputData['asset_type_id'], 'assets.asset_type_id');
        $assets     = $assets->relationData($inputData['user_id'], 'assets.user_id');
        $assets     = $assets->relationData($inputData['serial_no'], 'assets.serial_no');
        $assets     = $assets->relationData($inputData['location_id'], 'assets.location_id');
        $assets     = $assets->relationData($inputData['technical_spec_id'], 'assets.technical_spec_id');
        $assets     = $assets->relationData($inputData['asset_status_id'], 'assets.asset_status_id');
        $assets     = $assets->relationData($inputData['make_and_model_id'], 'assets.make_and_model_id');
        return $assets;
    }

    /**
     * Executing the query
     * @param Query $assets
     * @return Query Result
     */
    public function getOutputData($assets)
    {
        $start = request('start');
        $limit = request('length');
        if ($limit != -1) {
            $assets = $assets->skip($start)
                ->take($limit);
        }
        $assets = $assets->orderBy('id', 'desc');

        return $assets->get();
    }

    /**
     * Generating data table data from qurey result
     * @param $assets Query Result
     * @param $start Starting index
     * @param $data Collection valriable that will return
     * @return array
     */
    public function getReportData($assets, $start, $data = [])
    {
        $parentIndex = $start;
        foreach ($assets as $key => $asset) {
            $parentIndex++;
            $nestedData = $this->getNestedData($asset, $parentIndex);
            $data[] = $nestedData;
        }
        return $data;
    }

    /**
     * Generating data to export
     * @param $assets Query Result
     * @param $start Starting index
     * @param $data Collection valriable that will return
     * @return array
     */
    public function getReportExportData($assets, $start, $data)
    {
        $parentIndex = $start;

        foreach ($assets as $key => $asset) {
            $parentIndex++;
            $nestedData = $this->getExportNestedData($asset);
            $data[] = $nestedData['export'];
        }
        return $data;
    }

    /**
     * Gets the filter data values for dropdowns
     * @param mixed $type
     *
     * @return [type]
     */
    public function getFilterData($type)
    {
        return $this->repository->getFilterData($type);
    }

    /**
     * fetch intune filter values for report
     * @param string $type intune attribute name
     */
    public function getIntuneFilterData($type)
    {
        return IntuneComputer::select("$type")->groupBy($type)->orderBy($type)->get()->filter();
    }

    /**
     * fetch chromebook filter values for report
     * @param string $type chromebook attribute name
     */
    public function getChromebookFilterData($type)
    {
        return Chromebook::select("$type")->groupBy($type)->orderBy($type)->get()->filter();
    }
}
