<?php

namespace App\Services;

use Carbon\Carbon;
use App\Models\Asset;
use Cache;
use App\User;
use App\Models\Location;
use App\Models\EmployeeType;

class Notifications
{
    public function getOverDue($limit = '')
    {
        $notifications = $this->getAll();
        $itemCollection = collect($notifications);
        return $itemCollection->slice(0, $limit)->all();
    }

    public function getAll($limit = 8)
    {
        return Cache::remember('notification' . $limit, 360, function () {
            $allNotifications                               = [];
            $loanerPastDueNotifications                     = [];
            $retainHoldNotifications                        = [];
            $leaseTermEndNotifications                      = [];
            $ageOverFourYearsNotifications                  = [];
            $assignedInactiveUsersNotifications             = [];
            $activeInstallLocationVcNotifications           = [];
            $damagedAssetsNotifications                     = [];
            $activeFteInternAssetsNotifications             = [];
            $intransitLocationNotifications                 = [];

            $dateToday = Carbon::today()->toDateString();
            $lastWeek = Carbon::now()->addDays(7)->toDateString();


            foreach (Config('notification') as $type) {
                if ($type == 'loaner_over_due') {
                    if ($this->getLoanerPastDueNotifications()) {
                        $loanerPastDueNotifications = $this->getLoanerPastDueNotifications()->toArray();
                    }
                    $allNotifications = array_merge($allNotifications, $loanerPastDueNotifications);
                }

                if ($type == 'retain_hold_over_7_days') {
                    if ($this->getRetainHoldNotifications()) {
                        $retainHoldNotifications = $this->getRetainHoldNotifications()->toArray();
                    }
                    $allNotifications = array_merge($allNotifications, $retainHoldNotifications);
                }

                if ($type == 'lease_term_end') {
                    if ($this->getLeaseTermEndNotifications()) {
                        $leaseTermEndNotifications = $this->getLeaseTermEndNotifications()->toArray();
                    }
                    $allNotifications = array_merge($allNotifications, $leaseTermEndNotifications);
                }

                if ($type == 'asset_age_over_4_years') {
                    if ($this->getAgeOverFourYearsNotifications()) {
                        $ageOverFourYearsNotifications = $this->getAgeOverFourYearsNotifications()->toArray();
                    }
                    $allNotifications = array_merge($allNotifications, $ageOverFourYearsNotifications);
                }

                if ($type == 'assigned_to_inactive_users') {
                    if ($this->getAssignedInactiveUsersNotifications()) {
                        $assignedInactiveUsersNotifications = $this->getAssignedInactiveUsersNotifications()->toArray();
                    }
                    $allNotifications = array_merge($allNotifications, $assignedInactiveUsersNotifications);
                }

                if ($type == 'assigned_to_inactive_install_locations') {
                    if ($this->getAssignedInactiveInstallLocationsNotifications()) {
                        $assignedInactiveInstallLocationsNotifications = $this->getAssignedInactiveInstallLocationsNotifications()->toArray();
                    }
                    $allNotifications = array_merge($allNotifications, $assignedInactiveInstallLocationsNotifications);
                }

                if ($type == 'damaged_assets_over_30_days') {
                    if ($this->getDamagedAssetsNotifications()) {
                        $damagedAssetsNotifications = $this->getDamagedAssetsNotifications()->toArray();
                    }
                    $allNotifications = array_merge($allNotifications, $damagedAssetsNotifications);
                }

                if ($type == 'active_install_vc_location_without_asset(s)') {
                    if ($this->getActiveInstallLocationVcNotifications()) {
                        $activeInstallLocationVcNotifications = $this->getActiveInstallLocationVcNotifications()->toArray();
                    }
                    $allNotifications = array_merge($allNotifications, $activeInstallLocationVcNotifications);
                }

                if ($type == 'active_users_without_asset(s)') {
                    if ($this->getActiveFteInternAssetsNotifications()) {
                        $activeFteInternAssetsNotifications = $this->getActiveFteInternAssetsNotifications()->toArray();
                    }
                    $allNotifications = array_merge($allNotifications, $activeFteInternAssetsNotifications);
                }

                if ($type == 'assets_in_IN_transit_location') {
                    if ($this->getIntransitLocationNotifications()) {
                        $intransitLocationNotifications = $this->getIntransitLocationNotifications()->toArray();
                    }
                    $allNotifications = array_merge($allNotifications, $intransitLocationNotifications);
                }
            }

            return $allNotifications;
        });
    }

    public function getLoanerPastDueNotifications()
    {
        $dateToday = Carbon::today()->toDateString();
        $notifications = Asset::with('location', 'user')->hasStatus('loaned')
            ->whereDate('loaner_return_date', '<', $dateToday)->get();
        foreach ($notifications as $notification) {
            $notification->description = __('notification.LoanerPastDue', [
                'assetname' => $notification->serial_no,
                'assetid'   => $notification->id,
                'dueDate'   => $notification->loaner_return_date,
                'userName'  => optional($notification->user)->first_name . " " . optional($notification->user)->last_name . " (" . optional($notification->user)->email . ")"
            ]);
            $notification->type = 'loaner_over_due';
            $notification->site_code_id = ($notification->location_id != null) ? $notification->location?->site_code_id : null;
            $notification->region_id = ($notification->location_id != null) ? $notification->location?->region_id : null;
        }
        return $notifications;
    }

    public function getRetainHoldNotifications()
    {
        $lastWeek = Carbon::now()->subDays(7)->toDateString();
        $dateToday = Carbon::today()->toDateString();
        $notifications = Asset::with('location', 'user', 'assetStatus')->hasStatus('retain_hold')
            ->WhereDate('updated_at', '<', $lastWeek)->get();
        foreach ($notifications as $notification) {
            $updatedDate = date('Y-m-d', strtotime($notification->updated_at));

            $datediff = strtotime($dateToday) - strtotime($updatedDate);

            $difference = round($datediff / (60 * 60 * 24));

            $notification->description = __('notification.RetainHold', [
                'assetname' => $notification->serial_no,
                'assetid'   => $notification->id,
                'days'      => $difference,
                'userName'  => optional($notification->user)->first_name . " " . optional($notification->user)->last_name . " (" . optional($notification->user)->email . ")",
                'roomName'  => optional($notification->location)->room_name
            ]);
            $notification->type = 'retain_hold_over_7_days';
            $notification->site_code_id = ($notification->location_id != null) ? $notification->location?->site_code_id : null;
            $notification->region_id = ($notification->location_id != null) ? $notification->location?->region_id : null;
        }
        return $notifications;
    }

    public function getLeaseTermEndNotifications()
    {
        $dateToday = Carbon::today()->toDateString();
        $notifications = Asset::with('location', 'user')->notEndofLifeOrDisposed()
            ->where('lease_end_date', '<', $dateToday)->get();
        foreach ($notifications as $notification) {
            $status = ($notification->location_id != null) ? $notification->location?->room_name : ($notification->user_id != null ? $notification->user?->first_name . " " . $notification->user?->last_name . " (" . $notification->user?->email . ")" : " ");

            $notification->description = __('notification.LeaseTermEnd', [
                'assetname' => $notification->serial_no,
                'assetid'   => $notification->id,
                'dueDate'   => $notification->lease_end_date,
                'status'    => $status
            ]);
            $notification->type = 'lease_term_end';
            $notification->site_code_id = ($notification->location_id != null) ? $notification->location?->site_code_id : null;
            $notification->region_id = ($notification->location_id != null) ? $notification->location?->region_id : null;
        }
        return $notifications;
    }

    public function getAgeOverFourYearsNotifications()
    {
        $fourthLastYear = Carbon::now()->subYears(4)->toDateString();
        $notifications = Asset::with('location', 'user')->notEndofLifeOrDisposed()
            ->whereDate('created_at', '<', $fourthLastYear)->get();
        foreach ($notifications as $notification) {
            $status = ($notification->location_id != null) ? $notification->location?->room_name : ($notification->user_id != null ? $notification->user?->first_name . " " . $notification->user?->last_name . " (" . $notification->user?->email . ")" : " ");

            $notification->description = __('notification.AgeOverFourYears', [
                'assetname' => $notification->serial_no,
                'assetid'   => $notification->id,
                'status'    => $status
            ]);
            $notification->type = 'asset_age_over_4_years';
            $notification->site_code_id = ($notification->location_id != null) ? $notification->location?->site_code_id : null;
            $notification->region_id = ($notification->location_id != null) ? $notification->location?->region_id : null;
        }
        return $notifications;
    }

    public function getAssignedInactiveUsersNotifications()
    {
        $notifications = Asset::with('user', 'location')
            ->whereHas('user', function ($query) {
                $query->inactive();
            })->get();
        foreach ($notifications as $notification) {
            $notification->description = __('notification.AssignedInactiveUsers', [
                'assetname' => $notification->serial_no,
                'assetid'   => $notification->id,
                'userName'  => optional($notification->user)->first_name . ' ' . optional($notification->user)->last_name . " (" . optional($notification->user)->email . ")"
            ]);
            $notification->type = 'assigned_to_inactive_users';
            $notification->site_code_id = ($notification->location_id != null) ? $notification->location?->site_code_id : null;
            $notification->region_id = ($notification->location_id != null) ? $notification->location?->region_id : null;
        }
        return $notifications;
    }

    public function getAssignedInactiveInstallLocationsNotifications()
    {
        $notifications = Asset::with('location')
            ->whereHas('location', function ($query) {
                $query->inactive()->withoutWarehouse();
            })->get();
        foreach ($notifications as $notification) {
            $notification->description = __('notification.AssignedInactiveInstallLocation', [
                'assetname' => $notification->serial_no,
                'assetid'   => $notification->id,
                'roomName'  => optional($notification->location)->room_name
            ]);
            $notification->type = 'assigned_to_inactive_install_locations';
            $notification->site_code_id = ($notification->location_id != null) ? $notification->location?->site_code_id : null;
            $notification->region_id = ($notification->location_id != null) ? $notification->location?->region_id : null;
        }
        return $notifications;
    }

    public function getDamagedAssetsNotifications()
    {
        $lastMonth = Carbon::now()->subDays(30)->toDateString();
        $dateToday = Carbon::today()->toDateString();
        $notifications = Asset::with('location')->hasStatus('damaged')
            ->whereDate('updated_at', '<', $lastMonth)->get();
        foreach ($notifications as $notification) {
            $updatedDate = date('Y-m-d', strtotime($notification->updated_at));

            $datediff = strtotime($dateToday) - strtotime($updatedDate);

            $difference = round($datediff / (60 * 60 * 24));

            $notification->description = __('notification.DamagedAssets', [
                'assetname' => $notification->serial_no,
                'assetid'   => $notification->id,
                'days'      => $difference,
                'roomName'  => optional($notification->location)->room_name
            ]);
            $notification->type = 'damaged_assets_over_30_days';
            $notification->site_code_id = ($notification->location_id != null) ? $notification->location?->site_code_id : null;
            $notification->region_id = ($notification->location_id != null) ? $notification->location?->region_id : null;
        }
        return $notifications;
    }

    public function getActiveInstallLocationVcNotifications()
    {
        $notifications = Location::active()->withoutWarehouse()
            ->where('vc', 'VC')->doesnthave('assets')->get();
        foreach ($notifications as $notification) {
            $notification->description = __('notification.ActiveInstallLocationVc', [
                'roomName' => $notification->room_name
            ]);
            $notification->type = 'active_install_vc_location_without_asset(s)';
            $notification->site_code_id = $notification->site_code_id;
            $notification->region_id = $notification->region_id;
        }
        return $notifications;
    }

    public function getActiveFteInternAssetsNotifications()
    {
        $notifications = User::with('location')
            ->active()
            ->userRoles()
            ->doesnthave('assets')->get();
        foreach ($notifications as $notification) {
            $notification->description = __('notification.ActiveFteInternAssets', [
                'userName'     => $notification->first_name . ' ' . $notification->last_name . " (" . $notification->email . ")"
            ]);
            $notification->type = 'active_users_without_asset(s)';
            $notification->site_code_id = optional($notification->location)->site_code_id;
            $notification->region_id = optional($notification->location)->region_id;
        }
        return $notifications;
    }

    public function getIntransitLocationNotifications()
    {
        $dateToday = Carbon::today()->toDateString();
        $notifications = Asset::with('assetType', 'makeAndModel', 'technicalSpec')
            ->whereHas('location', function ($query) {
                $query->where('room_name', 'like', '%TRANSIT%');
            })
            ->whereDate('updated_at', '<=', Carbon::now()->subDays(15)->toDateString())->get();
        foreach ($notifications as $notification) {
            $updatedDate = date('Y-m-d', strtotime($notification->updated_at));

            $datediff = strtotime($dateToday) - strtotime($updatedDate);

            $difference = round($datediff / (60 * 60 * 24));

            $notification->description = __('notification.IntransitLocations', [
                'assetname' => $notification->serial_no,
                'assetid'   => $notification->id,
                'roomName'  => optional($notification->location)->room_name,
                'days'      => $difference
            ]);
            $notification->type = 'assets_in_IN_transit_location';
            $notification->site_code_id = ($notification->location_id != null) ? $notification->location?->site_code_id : null;
            $notification->region_id = ($notification->location_id != null) ? $notification->location?->region_id : null;
        }

        return $notifications;
    }
}
