<?php

namespace App\Services\NewHire;

use Carbon\Carbon;
use App\Repositories\NewHire\ForecastingToolRepository as ForecastingToolRepo;

class NewHireForecastingToolData
{
    protected $forecastingToolRepo;

    public function __construct(ForecastingToolRepo $forecastingToolRepo)
    {
        $this->forecastingToolRepo = $forecastingToolRepo;
    }


    public function filter()
    {
        $newhireForecastingData =  $this->forecastingToolRepo->getForcastingData();
        $newhireForecastingData =  $this->filterWithInputData($newhireForecastingData);
        $forecastingData = $newhireForecastingData->get()->toArray();
        // dd($newhireForecastingData->toSql());
        $count = $newhireForecastingData->get()->count();
        return compact('forecastingData', 'count');
    }

    /**
     * It filters the data based on the input data
     *
     * @param newhireForecastingData This is the data that you want to filter.
     */
    public function filterWithInputData($newhireForecastingData)
    {
        $inputData = $this->getInputData();
        $newhireForecastingData =  $this->forecastingToolRepo->filterWithDepartment($newhireForecastingData, $inputData['department']);
        $newhireForecastingData =  $this->forecastingToolRepo->filterWithHireMonthRange($newhireForecastingData, $this->convertMonthToDate($inputData['start_month']), $this->convertMonthToDate($inputData['end_month']));
        return $newhireForecastingData;
    }

    public function createNewhireForecastingCollection($finalData)
    {
        $finalDataCollection = collect($finalData);

        $result = $finalDataCollection->groupBy(function ($item, $key) {
            return $item['department'];
        });

        return  $result->toArray();
    }

    /**
     * Generating data table data from query result
     * @param $data Collection valriable that will return
     * @return array
     */
    public function getData($data, $forecastingData)
    {
        foreach ($forecastingData as $forecastingDataRow) {
            $nestedData = $this->getNestedData($forecastingDataRow);
            $data[] = $nestedData;
        }
        $result = $this->createNewhireForecastingCollection($data);

        return $result;
    }
    /**
     * It takes a row of data from a database table, and returns a new array with some of the data from
     * the original row, and some new data that is calculated from the original data
     *
     * @param forecastingDataRow This is the row of data that is being processed.
     */
    public function getNestedData($forecastingDataRow)
    {
        $departmentId = isset($forecastingDataRow['department_id']) ? $forecastingDataRow['department_id'] : '';
        $data['hardware_standard']  = isset($forecastingDataRow['make_and_model']['name']) ? $forecastingDataRow['make_and_model']['name'] : '';
        $data['tech_spec']          = isset($forecastingDataRow['technical_spec']['details']) ? $forecastingDataRow['technical_spec']['details'] : '';
        $data['department']  = isset($forecastingDataRow['single_department']['name']) ? $forecastingDataRow['single_department']['name'] : '';
        $data['quantity_needed']    = $this->getQuantityNeeded($departmentId);
        $data['avg_value']          = "$ " . $forecastingDataRow['cost'];
        $data['cost']               = "$ " . round($data['quantity_needed'] * $forecastingDataRow['cost']);
        $data['cost_cal']           = round($data['quantity_needed'] * $forecastingDataRow['cost']);
        $data['quantity_reserved']  = $forecastingDataRow['user_count'];
        $data['quantity_available'] = $forecastingDataRow['in_stock_assets_count'];
        $data['purchase_indicator'] = ($data['quantity_available'] - $data['quantity_reserved']) >= $data['quantity_needed'] ? "No" : "Yes";
        if ($data['quantity_available'] < 0) {
            $data['quantity_available'] = 0;
        }
        return $data;
    }

    /**
     * It takes a department ID and returns the number of new hires needed for that department
     *
     * @param departmentId The department id of the department you want to get the quantity needed for.
     *
     * @return The quantity needed for a department.
     */
    public function getQuantityNeeded($departmentId)
    {
        $inputData = $this->getInputData();
        $departments = $inputData['department'];
        $quantityNeeded = 0;
        if (in_array($departmentId, $departments)) {
            $arrayKey = array_search($departmentId, $departments);
            $quantityNeeded = isset($inputData['newhire'][$arrayKey]) ? $inputData['newhire'][$arrayKey] : 0;
        }
        return $quantityNeeded;
    }


    /**
     * It returns an array of the request data, but if the request data is empty, it returns the old
     * input data
     *
     * @return The input data is being returned.
     */
    public function getInputData()
    {
        return [
            'newhire'         => request('new_hires') ?? request('new_hires'),
            'department'      => request('department') ?? request('department'),
            'start_month'     => request('start_month_year') ?? request('start_month_year'),
            'end_month'       => request('end_month_year') ?? request('end_month_year'),
        ];
    }

    /**
     * It takes a month number and returns a Carbon object representing the first day of that month in
     * the next year
     *
     * @param month The month you want to get the data for.
     *
     * @return The date of the first day of the month.
     */
    public function convertMonthToDate($monthYear, $returnType = null)
    {
        if ($monthYear != null) {
            $monthYear = Carbon::createFromFormat('M-Y', $monthYear);
            $month = $monthYear->format('m');
            $year = $monthYear->format('Y');
            if (valid_date_or_not($month . '/01/' . $year)) {
                $dateString = $year . '-' . $month . '-01';
                return ($returnType == 'month') ? $monthYear->format('F') : new  Carbon($dateString);
            }
            return null;
        }
        return null;
    }
}
