<?php

namespace App\Services\Location;

use App\Http\Responses\ReportOutputData;
use App\Repositories\CommonFilterRepository;
use App\Repositories\LocationRepository;

class LocationZoomDeviceService
{
    /**
     * Constructor
     *
     * @param LocationRepository $repository The repository for locations.
     * @param CommonFilterRepository $commonFilterRepository The repository for common filters.
     * @param ReportOutputData $reportOutputData
     */
    public function __construct(
        protected LocationRepository $repository,
        protected CommonFilterRepository $commonFilterRepository,
        protected ReportOutputData $reportOutputData
    ) {}

    /**
     * Gets the zoom devices for a location report.
     *
     * @return array Array containing the assets and their count
     */
    public function data()
    {
        $locationId = request('form')['location'] ?? null;
        $serialNo   = request('form')['serial_no'] ?? '';
        $assets     = $this->repository->getZoomDevices($locationId);
        $assets     = $this->commonFilterRepository->filterWithWhereHasRelationLikeFields($assets, 'asset', 'serial_no', $serialNo);
        $count      = $assets->count();
        $assets     = $this->reportOutputData->getOutputData($assets, ['room_name' => 'asc']);

        return compact('assets', 'count');
    }

    /**
     * Gets the data table for a zoom assets list in the location report.
     *
     * @param Collection $assets Zoom asset data
     * @param int $start Starting index
     * @return array Nested data
     */
    public function getLocationZoomData($assets, $start)
    {
        $data = [];
        $parentIndex = $start;

        foreach ($assets as $singleRecord) {
            $parentIndex++;
            $nestedData = $this->getNestedLocationZoomData($singleRecord, $parentIndex);
            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Generates the nested data for a zoom assets list in the location report.
     *
     * @param mixed $zoom Zoom asset data
     * @param int $index Starting index
     * @return array Nested data
     */
    public function getNestedLocationZoomData($zoom, $index)
    {
        $nestedData['id']           = $index;
        $nestedData['serial_no']    = $zoom->asset ? generateAssetLink($zoom->asset_id, $zoom->serial_no) : $zoom->serial_no;
        $nestedData['checkbox']     = $this->getZoomInstalledAssetCheckbox($zoom);

        return $nestedData;
    }

    /**
     * Generates a checkbox for a zoom asset in the location report.
     * If the zoom asset is already installed in the location, it will show a marked checkbox.
     * Otherwise, it will be empty.
     *
     * @param mixed $zoom Zoom asset data
     * @return string Checkbox HTML
     */
    public function getZoomInstalledAssetCheckbox($zoom)
    {
        if (!$zoom->asset) {
            return '';
        }

        if ($zoom->location_id === optional($zoom->asset)->location_id) {
            return '<input type="checkbox" name="zoomAsset" value="' . $zoom->id . '" checked disabled>';
        }

        return '';
    }
}
