<?php

namespace App\Services\Location;

use App\Http\Responses\ReportOutputData;
use App\Repositories\LocationRepository;
use Illuminate\Support\Str;

class LocationData
{

    public function __construct(protected LocationRepository $locationRepository, protected  ReportOutputData $reportOutputData) {}

    /**
     * Queries and returns the locations after filtering
     * @return [type]
     */
    public function getData()
    {
        $locations = $this->locationRepository->getLocations();

        $inputData = $this->getInputData(request('form'));

        $locations = $this->filter($locations, $inputData);

        $count = $locations->count();

        $locations = $this->reportOutputData->getOutputData($locations, ['room_name' => 'asc']);

        return compact('locations', 'count');
    }

    /**
     * Returns the input form data
     * @param mixed $formData
     *
     * @return [type]
     */
    private function getInputData($formData)
    {
        return [
            'searchText'   => $formData['searchText'] ??  '',
            'locationType' => $formData['location_type'] ??  '',
            'haveAssets'   => $formData['have_assets'] ??  '',
            'noOfAssets'   => $formData['no_of_assets'] ??  '',
            'status'       => $formData['status'] ??   '',
            'country'      => $formData['country'] ??  '',
            'region'       => $formData['region'] ??  '',
            'vc'           => $formData['vc'] ??  '',
            'zoom'         => $formData['zoom'] ?? '',
        ];
    }

    /**
     * Carries the filter opertaions
     * @param mixed $locations
     * @param mixed $inputData
     *
     * @return [type]
     */
    private function filter($locations, $inputData)
    {

        $locations = $this->locationRepository->filterLocationFields($locations, 'location_type_id', $inputData['locationType']);
        $locations = $this->locationRepository->filterLocationFields($locations, 'country_id', $inputData['country']);
        $locations = $this->locationRepository->filterLocationFields($locations, 'region_id', $inputData['region']);
        $locations = $this->locationRepository->searchLocationByKeyword($locations, $inputData['searchText']);
        $locations = $this->locationRepository->filterByLocationStatus($locations, $inputData['status']);
        $locations = $this->locationRepository->filterLocationHasAssets($locations, $inputData['haveAssets']);
        $locations = $this->locationRepository->filterLocationByNoofAssets($locations, $inputData['noOfAssets']);
        $locations = $this->locationRepository->filterLocationByVc($locations, $inputData['vc']);
        $locations = $this->locationRepository->filterZoom($locations, $inputData['zoom']);

        return $locations;
    }

    /**
     * Makes array for report response
     * @param mixed $locations
     * @param mixed $start
     * @param mixed $data
     *
     * @return [type]
     */
    public function getLocationData($locations, $start, $data)
    {
        $parentIndex = $start;

        foreach ($locations as $key => $location) {

            $parentIndex++;

            $name = route('location.show', $location->id);

            $nestedData = $this->getNestedData($location, $parentIndex, '', '', $name);

            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Makes array of individual rows for the report
     * @param mixed $location
     * @param mixed $index
     * @param mixed $last
     * @param mixed $parent
     * @param mixed $name
     *
     * @return [type]
     */
    public function getNestedData($location, $index, $last, $parent, $name)
    {
        $nestedData = [];
        $commonData = $this->getExportNestedData($location);
        $nestedData['id']   = $index;
        $nestedData['dataLast']     = $last;
        $nestedData['dataId']       = $location->id;
        $nestedData['dataParent']   = $parent;
        $nestedData['room_name']         = "<a href=$name>$location->room_name</a>";
        foreach ($commonData as $key => $value) {
            $nestedData[Str::slug($key, '_')] = $value;
        }
        $nestedData['action'] = "<a class='btn btn-link' href=location/" . $location->id . "/edit><i class='icon icon-n-edit'></i></a>";

        return $nestedData;
    }


    /**
     * Making query for export
     */
    public function getExportData($requestData = array())
    {
        if (!empty($requestData)) {
            request()->merge($requestData);
        }
        $locations = $this->locationRepository->getLocations();
        $inputData = $this->getInputData(request()->all());
        $locations = $this->filter($locations, $inputData);
        $locations = $locations->orderBy('room_name', 'asc');
        return $locations;
    }

    /**
     * Making individual row array for export
     * @param mixed $locations
     * @param mixed $data
     *
     * @return [type]
     */
    public function getLocationExportData($locations)
    {
        $data = [];
        foreach ($locations as $key => $location) {
            $nestedData = $this->getExportNestedData($location);
            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Makes array for export data
     * @param mixed $location
     *
     * @return [type]
     */
    public function getExportNestedData($location)
    {
        $nestedData['Site Code']    = optional($location->siteCode)->code;
        $nestedData['Type']         = optional($location->locationType)->location_type_name;
        $nestedData['Name']         = $location->room_name;
        $nestedData['Address']      = $location->address;
        $nestedData['Floor']        = $location->floor;
        $nestedData['Building']     = $location->building;
        $nestedData['Country']      = optional($location->country)->name;
        $nestedData['Region']       = optional($location->region)->name;
        $nestedData['Vc']           = $location->vc;
        $nestedData['Capacity']     = $location->room_capacity;
        $nestedData['Zoom']             = disableCSVInjection($location->zoom);
        $nestedData['Status']       = $location->status == 1 ? 'Active' : 'Inactive';
        $nestedData['# of Assets'] = $location->assets_count;
        return $nestedData;
    }
}
