<?php

namespace App\Services;

use App\Models\ApiCredential;
use Facades\App\Services\Integrations\Tickets\Jira\OAuthWrapper;
use Illuminate\Support\Facades\Log;

class JiraService
{
    public function __construct()
    {

    }

    /**
     * Returns the token
     * @return [type]
     */
    private function getToken()
    {
        $jiraCredential = ApiCredential::where('slug', 'jira')->first();

        return [
            'oauth_token' => optional($jiraCredential)->project,
            'oauth_token_secret' => optional($jiraCredential)->key
        ];
    }

    public function sendToJira($title = '', $description = '', $ticketId, $userId)
    {
        try {
            $user = OAuthWrapper::getAuthorizedUser($this->getToken());
            $status = 1;
        } catch (\Exception $e) {
            $status = 0;
        }
        if ($status == 1) {
            if (!$description) {
                return false;
            }

            $issue = OAuthWrapper::addComment($this->getToken(), $ticketId, $description, $userId);

            return !empty($issue->id) ? $issue->id : null;
        }
        return true;
    }

    public function checkTicketExists($ticketId)
    {
        try {
            $user = OAuthWrapper::getAuthorizedUser($this->getToken());
            $status = 1;
        } catch (\Exception $e) {
            $status = 0;
        }
        if ($status == 1) {
            try {
                $response = OAuthWrapper::getIssueId($this->getToken(), $ticketId);
                if ($response['fields']['status']['name'] == "Done") {
                    return false;
                }
                return true;
            } catch (\Exception $e) {
                return false;
            }
        }
        return true;
    }

    public function bulkSendToJira($jiraData)
    {
        //Bulk Update Jira Ticket by Queuying
        if ($jiraData) {
            $jiraDataCollection = collect($jiraData);
            $jiraDatas = $jiraDataCollection->chunk(25);
            foreach ($jiraDatas as $jiraData) {
                event(new BulkUpdateTicket($jiraData));
            }
        }
    }

    /**
     * Take jira ticket detail by using label and custom field
     *
     * @param string $label              Lable key
     * @param string $lablevalue         Lable Value
     * @param string $customField        Custom Field Name
     * @param string $customFieldValue   Custom Field value
     *
     * @return array
     */
    public function getIssueByUserEmailAndComponent($email)
    {
        $response = [];
        try {
            $user = OAuthWrapper::getAuthorizedUser($this->getToken());
            $status = 1;
        } catch (\Exception $e) {
            $status = 0;
        }
        if ($status == 1) {
            try {
                $response = OAuthWrapper::getIssueByUserEmailAndComponent($this->getToken(), $email);
                return $response;
            } catch (\Exception $e) {
                return $response;
            }
        }
        return $response;
    }

    /**
     * List out all custom fields details
     *
     * @param null
     *
     * @return boolean
     */
    public function getAllCustomFields()
    {
        try {
            $user = OAuthWrapper::getAuthorizedUser($this->getToken());
            $status = 1;
        } catch (\Exception $e) {
            $status = 0;
        }
        if ($status == 1) {
            try {
                $response = OAuthWrapper::allCustomFields($this->getToken());
                return $response;
            } catch (\Exception $e) {
                return false;
            }
        }
        return true;
    }

    public function createTicket($data)
    {
        try {
            $user = OAuthWrapper::getAuthorizedUser($this->getToken());
            $status = 1;
        } catch (\Exception $e) {
            $status = 0;
            Log::channel('daily')->info('Jira Ticket Create Error :  ' . $e->getMessage());
        }
        if ($status == 1) {
            $issue = OAuthWrapper::createTicket($this->getToken(), $data);

            return !empty($issue->id) ? $issue->id : null;
        }
        return true;
    }

    public function getIssueStatus(string $issueId): string
    {
        try {
            $user = OAuthWrapper::getAuthorizedUser($this->getToken());
            $status = 1;
        } catch (\Exception $e) {
            $status = 0;
        }

        if ($status == 1) {
            try {
                $response = OAuthWrapper::getIssueCurrentStatus($this->getToken(), $issueId);
                return !empty($response) ? $response['fields']['status']['name'] : "";
            } catch (\Exception $e) {
            }
        }
        return "";
    }

    /**
     * Fetch tickets by user status (active user component : Laptop Provisioning, terminated user component : Laptop Deprovisioning)
     */
    public function getTicketsByUserStatus($status, $index, $limit)
    {
        $customEmailField = config('jira-fields.sync.custom_email_field');
        $customStartDate = config('jira-fields.sync.custom_start_date');
        $customEndDate = config('jira-fields.sync.terminated_date');
        $newHire = config('jira-fields.sync.new_hire_issue_type');
        $terminated = config('jira-fields.sync.terminated_user_issue_type');

        if ($status == 0) {
            $url = "rest/api/2/search?jql=component='" . $terminated . "'&fields=id,key,status,component," . $customStartDate . "," . $customEndDate . "," . $customEmailField . "," . "&startAt=" . $index . "&maxResults=" . $limit;
        } else {
            $url = "rest/api/2/search?jql=component='" . $newHire . "' &fields=id,key,status,component," . $customEmailField . "," . "&startAt=" . $index . "&maxResults=" . $limit;
        }

        $guzzle = OAuthWrapper::getClient($this->getToken()['oauth_token'], $this->getToken()['oauth_token_secret']);

        $response = $guzzle->get($url)->getBody()->getContents();
        return json_decode($response, true);
    }

    /**
     * This function is used to get the hardware request ticket details(Asset refresh  and replacement request) by component type
     *
     * @param component The component type you want to search for.(laptop refresh or laptop replacement)
     * @param index The starting index of the search results.
     * @param limit The number of results you want to return.
     *
     * @return An array of JIRA issues that match the criteria.
     */
    public function getTicketDetailsByComponent($component, $index, $limit)
    {
        $oldAsset = config('jira-fields.old_asset_serial_asset_tag');
        $newAsset = config('jira-fields.new_asset_serial_asset_tag');
        $requestDate = config('jira-fields.refresh_replacement_date');
        $url = "rest/api/2/search?jql=component='" . $component . "'&fields=id,key,status,component," . $oldAsset . "," . $newAsset . "," . $requestDate . "," . "&startAt=" . $index . "&maxResults=" . $limit;
        $guzzle = OAuthWrapper::getClient($this->token['oauth_token'], $this->token['oauth_token_secret']);
        $response = $guzzle->get($url)->getBody()->getContents();

        return json_decode($response, true);
    }
}
