<?php

namespace App\Services\Integrations\UserDirectory;

use App\User;
use App\Models\UserType;
use App\Models\ApiCredential;
use Illuminate\Support\Facades\Log;
use App\Repositories\HiBobRepository;
use Illuminate\Support\Facades\Http;

class HiBobIntegration
{

    /**
     * Construct a new HiBobIntegration instance.
     * 
     * @param HiBobRepository $hiBobRepository The HiBob API repository instance.
     */
    public function __construct(protected HiBobRepository $hiBobRepository) {}

    /**
     * HiBob API credentials
     */
    public function getApiCredentials()
    {
        return ApiCredential::where('slug', 'hibob')->first();
    }

    /**
     * Check the API connection
     */
    public function ping()
    {
        try {
            $response = $this->callGetApi('company/people/fields');
            if ($response->getStatusCode() == 200) {
                return true;
            } else {
                return false;
            }
        } catch (\Exception $e) {
            Log::info($e->getMessage());
            return false;
        }
    }

    /**
     * Make a GET request to the HiBob API
     * 
     * @param string $endpoint The API endpoint to call
     * @return \Psr\Http\Message\ResponseInterface
     */
    private function callGetApi($endpoint)
    {
        $username = $this->getApiCredentials()->user_name ?? '';
        $password = $this->getApiCredentials()->password ?? '';

        return Http::withBasicAuth($username, $password)->get('https://api.hibob.com/v1/' . $endpoint);
    }

    /**
     * Call the API with POST method
     * 
     * @param string $endpoint API Endpoint
     * @param array $data The data to be sent with the request
     * @return \Psr\Http\Message\ResponseInterface
     */
    private function callPostApi($endpoint, array $data = [])
    {
        $username = $this->getApiCredentials()->user_name ?? '';
        $password = $this->getApiCredentials()->password ?? '';

        return Http::withBasicAuth($username, $password)->post('https://api.hibob.com/v1/' . $endpoint, $data);
    }

    /**
     * Sync All users from the HiBob Directory
     */
    public function syncAllUsers()
    {
        $count = 0;
        $users = $this->getAllUsers();

        if ($users) {
            foreach ($users as $user) {
                $count += $this->saveUser($user);
            }
        }
        return $count;
    }

    /**
     * Get all users from HiBob API
     */
    public function getAllUsers()
    {
        try {
            $url = 'people/search';

            $response = $this->callPostApi($url, [
                'showInactive'          => true,    // include inactive users
                'fields'                => [       // request specific fields
                    'root.id',
                    'root.firstName',
                    'root.surname',
                    'root.email',
                    "work.title",
                    'work.department',
                    'work.employeeIdInCompany',
                    'work.site',
                    'work.startDate',
                    'work.custom.field_1685044954183',
                    'internal.terminationDate',
                    'payroll.employment.type',
                    'root.creationDateTime',
                    'work.reportsToIdInCompany',
                    'internal.status',
                    'internal.lifecycleStatus',
                ],
                'humanReadable'  => 'APPEND',
            ]);

            $responseData = $response->object();
            return $responseData->employees ?? null;
        } catch (\Exception $e) {
            Log::error('HiBob API request failed', [
                'method' => __METHOD__,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    /**
     * Save the use details to DB
     * 
     * @param $item  API response user data
     */
    public function saveUser($item)
    {
        $email = $item->email ?? null;

        if (empty($email)) {
            return 0;
        }

        // Prepare base user data
        $userData = $this->generateUserData($item);

        // If user doesn't exist, assign default user type
        $existingUser = User::where('email', $email)->first();
        if (!$existingUser) {
            $userData['user_type_id'] = UserType::people()->first()->id ?? null;
        }

        // Create or update the user
        User::updateOrCreate(['email' => $email], $userData);

        return 1;
    }

    /**
     * Maps the API response data to the corresponding database fields.
     *
     * @param object $item HiBob API response user data.
     * @return array|false An associative array with mapped user data, or false if the input item is empty.
     */
    public function generateUserData($item)
    {
        $headerMap = config('hibob-directory.userData');

        if (empty($item)) {
            return false;
        }

        foreach ($headerMap as $dbField => $fileHeader) {
            $result = $this->getRelationalValues($item, $dbField);

            if ($result !== false) {
                $item->$fileHeader = $result;
            }
            $userData[$dbField] = isset($item->$fileHeader) ? $item->$fileHeader : null;
        }

        return $userData;
    }

    /**
     * Maps the HiBob API response data to the corresponding relational database fields.
     * The method takes the API response user data and the database field name as arguments.
     * It uses a lookup table to determine which HiBobRepository method to call for the given field.
     * The method returns the value returned by the HiBobRepository method, or false if the input item is empty.
     * 
     * @param object $item HiBob API response user data.
     * @param string $dbField The database field name.
     * 
     * @return mixed The value returned by the HiBobRepository method, or false if the input item is empty.
     */
    public function getRelationalValues($item, $dbField)
    {
        $status = $this->hiBobRepository->getStatus($item->internal->status ?? null);

        $map = [
            'position_id'       => fn() => $this->hiBobRepository->getPositionId($item->humanReadable->work->title ?? null),
            'department_id'     => fn() => $this->hiBobRepository->getDepartmentId($item->humanReadable->work->department ?? null),
            'country_id'        => fn() => $this->hiBobRepository->getCountryId($item->humanReadable->work->custom->field_1685044954183 ?? null),
            'city'              => fn() => $item->humanReadable->work->site ?? null,
            'employee_type_id'  => fn() => $this->hiBobRepository->getEmployeeTypeId($item->payroll->employment->contract ?? null),
            'status'            => fn() => $status,
            'manager_id'        => fn() => $this->hiBobRepository->getManagerId($item->work->reportsToIdInCompany),
            'employee_id'       => fn() => $item->work->employeeIdInCompany ?? null,
            'hire_date'         => fn() => parse_hibob_date($item->work->startDate ?? null),
            'terminated_date'   => fn() => ($status === 0) ? parse_hibob_date($item->internal->terminationDate ?? null) : null,
            'created_at'        => fn() => parse_hibob_date($item->creationDateTime ?? null),
        ];

        return isset($map[$dbField]) ? $map[$dbField]() : false;
    }
}
