<?php

namespace App\Services\Integrations\Tickets;

use App\Events\BulkUpdateTicket;
use App\Services\Integrations\Tickets\TicketIntegrationInterface;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

/**
 * Service class manages ticket services
 */
class TicketManagementService
{
    protected  $ticketIntegration;
    /**
     * __construct
     *
     * @param  TicketIntegrationInterface $ticketIntegration
     * @return void
     */
    public function __construct(TicketIntegrationInterface $ticketIntegration)
    {
        $this->ticketIntegration = $ticketIntegration;
    }

    /**
     * addComment to existing ticket
     *
     * @param  string $title
     * @param  string $description
     * @param  int $ticketId
     * @param  int $userId
     * @return int
     */
    public function addComment($userId, $ticketId, $description = '', $title = '')
    {

        if (!$description || !$ticketId) {
            return false;
        }
        try {
            $comment = $this->ticketIntegration->addCommentToTicket($ticketId, $description, $userId);
            return !empty($comment->id) ? $comment->id : null;
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * create new ticket
     *
     * @param  array $ticketData
     * @return void
     */
    public function createTicket($ticketData)
    {
        $this->ticketIntegration->createTicket($ticketData);
    }

    /**
     * Bulk Update Ticket comments 
     *
     * @param  array $ticketData
     * @return void
     */
    public function bulkAddComment($ticketData)
    {
        if ($ticketData) {
            $ticketDataCollection = collect($ticketData);
            $ticketDatas = $ticketDataCollection->chunk(25);
            foreach ($ticketDatas as $ticketData) {
                event(new BulkUpdateTicket($ticketData));
            }
        }
    }

    /**
     * check if the ticket is already existing
     *
     * @param  int $ticketNo
     * @return boolean
     */
    public function checkTicketExists($ticketNo)
    {
        if ($ticketNo) {
            try {
                return (bool) $this->ticketIntegration->getTicket($ticketNo);
            } catch (\Exception $e) {
                return false;
            }
        }
        return false;
    }

    /**
     * attaches the action history here the bulk upload file is attached to ticket
     *
     * @param  int $ticketNo
     * @return boolean
     */
    public function attachActionHistorytoTicket($ticketNo, $actionHistory, $userId, $filepath, $fileName)
    {
        if ($ticketNo) {
            return (bool) $this->ticketIntegration->addAttachmentToTicket($ticketNo, $actionHistory, $userId, $filepath, $fileName);
        }
        return false;
    }

    /**
     * Get the ticket url dynamically based on current ticket service using
     * @param mixed $ticketNo
     * 
     * @return [type]
     */
    public function getTicketUrl($ticketNo)
    {
        return $this->ticketIntegration->getTicketUrl($ticketNo);
    }

    /**
     * Retrieves the URL of a historical ticket based on the ticket integration and ticket number.
     *
     * @param string $ticketIntegration The ticket integration (e.g. 'Jira', 'Zendesk', 'FreshService', 'ServiceNow').
     * @param string $ticketNo The ticket number.
     * @throws Exception If an error occurs while retrieving the ticket URL.
     * @return string The URL of the historical ticket.
     */
    public function getHistoryTicketUrl($ticketIntegration = 'Jira', $ticketNo)
    {
        $ticketIntegrationClass = match ($ticketIntegration) {
            'Jira' => \App\Services\Integrations\Tickets\Jira\JiraIntegration::class,
            'Zendesk' => \App\Services\Integrations\Tickets\Zendesk\ZendeskIntegration::class,
            'FreshService' => \App\Services\Integrations\Tickets\FreshService\FreshServiceIntegration::class,
            'ServiceNow' => \App\Services\Integrations\Tickets\ServiceNow\ServiceNowIntegration::class,
            default => \App\Services\Integrations\Tickets\Jira\JiraIntegration::class,
        };

        $integrationInstance = new $ticketIntegrationClass;

        $url = $integrationInstance->getTicketUrl($ticketNo);

        return $url;
    }
}
