<?php

namespace App\Services\Integrations\Tickets\Jira;

use Facades\App\Services\Integrations\Tickets\Jira\OAuthWrapper;
use App\Models\ApiCredential;
use App\Models\FailedTicketHistory;
use App\Services\Integrations\Tickets\TicketIntegrationInterface;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;

class JiraIntegration implements TicketIntegrationInterface
{
    protected $client;

    /**
     * Construct for JiraIntegration class
     */
    public function __construct()
    {
    }

    /**
     * Retrieves the API credentials for Jira from the database.
     *
     * @return ApiCredential|null The API credentials for Jira, or null if not found.
     */
    private function getCredentials()
    {
        return ApiCredential::where('slug', 'jira')->first();
    }

    /**
     * Retrieves the OAuth token and token secret from the API credentials for Jira.
     *
     * @return array An associative array containing the OAuth token and token secret.
     */
    private function getToken()
    {
        $credential = $this->getCredentials();
        return [
            'oauth_token' => optional($credential)->project ?? "",
            'oauth_token_secret' => optional($credential)->key ?? ""
        ];
    }

    /**
     * Get the ticket details by ID
     * @param string $requestId
     * 
     * @return [type]
     */
    public function getTicket(string $requestId)
    {
        $issue = OAuthWrapper::getIssueId($this->getToken(), $requestId);
        return $issue;
    }

    /**
     * Creates new ticket from $ticketdata
     * @param array $ticketData
     * 
     * @return [type]
     */
    public function createTicket(array $ticketData)
    {
        return OAuthWrapper::createTicket($this->getToken(), $ticketData);
    }

    /**
     * Adds message to existing ticket
     * @param string $requestId
     * @param string $message
     * @param string $userId
     * 
     * @return [type]
     */
    public function addCommentToTicket(string $requestId, string $message, string $userId)
    {
        if (!$message) {
            return false;
        }
        try {
            OAuthWrapper::getAuthorizedUser($this->getToken());
            $issue = OAuthWrapper::addComment($this->getToken(), $requestId, $message, $userId);
            return !empty($issue['id']);
        } catch (\Exception $e) {
            FailedTicketHistory::create([
                'user_id' => $userId,
                'ticket_id' => $requestId,
                'description' => $message, //str_replace("\\n", '', $message),
                'exceptions' => json_encode($e->getMessage())
            ]);
            return false;
        }
    }

    /**
     * Add an attachment to an existing ticket
     * @param string $requestId
     * @param string $message
     * @param string $userId
     * @param mixed $file
     * @param mixed $fileName
     * 
     * @return [type]
     */
    public function addAttachmentToTicket(string $requestId, string $message, string $userId, $file, $fileName)
    {
        if (!$message || !$fileName) {
            return false;
        }
        $description = $message . '. Please refer attachment ' . $fileName . ' for details';
        try {
            OAuthWrapper::getAuthorizedUser($this->getToken());
            if (OAuthWrapper::addAttachmentAsComment($this->getToken(), $requestId, $userId, $file, $fileName)) {
                $issue = OAuthWrapper::addComment($this->getToken(), $requestId, $description, $userId);
                return !empty($issue['id']);
            }
        } catch (\Exception $e) {
            FailedTicketHistory::create([
                'user_id' => $userId,
                'ticket_id' => $requestId,
                'description' => str_replace("\\n", '', $description),
                'exceptions' => json_encode($e->getMessage())
            ]);
            return false;
        }
    }
    /**
     * Get the complete ticket url
     * @param string $requestId
     * 
     * @return [type]
     */
    public function getTicketUrl($requestId)
    {
        // Cache the Jira URL for 60 minutes (adjust the time as needed).
        $jiraUrl = Cache::remember('jira_credential_url', 3600, function () {
            return optional($this->getCredentials())->url;
        });

        if ($jiraUrl) {
            // Example URL: https://teqube.atlassian.net/browse/TEQ-160
            return $requestId ? '<a href="' . $jiraUrl . 'browse/' . $requestId . '" target="_blank">' . $requestId . '</a>' : '';
        }

        return '';
    }
}
