<?php

namespace App\Services\Integrations\Tickets\FreshService;

use App\Models\ApiCredential;
use App\Services\Integrations\Tickets\TicketIntegrationInterface;
use Exception;
use Illuminate\Support\Facades\Http;
use App\Models\FailedTicketHistory;
use Google\Service\ServiceControl\Auth;
use GuzzleHttp\Client;
use GuzzleHttp\Psr7;
use Illuminate\Support\Facades\Cache;

/**
 * FreshServiceIntegration
 */
class FreshServiceIntegration implements TicketIntegrationInterface
{

    public function getCredentials()
    {
        $freshServiceCredential = ApiCredential::where('slug', 'freshservice')->first();
        $credentials = [
            'baseUrl' => $freshServiceCredential ? $freshServiceCredential->url . 'api/v2/' : '',
            'apiKey' => $freshServiceCredential ? $freshServiceCredential->key : '',
            'password' => 'password',
            'freshServiceCredential' => $freshServiceCredential
        ];
        return (object) $credentials;
    }


    /**
     * Get Guzzle client
     * @param null $clientId
     * @param null $clientSecret
     * 
     * @return [type]
     */
    public function getClient($clientId = null, $clientSecret = null)
    {
        $credentials = $this->getCredentials();
        $this->client = new Client([
            'base_uri' => $credentials->baseUrl,
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'Authorization' => 'Basic ' . base64_encode($clientId . ':' . $clientSecret),
            ],
        ]);

        return $this->client;
    }

    /**
     * createTicket
     *
     * @param  array $ticketData
     * @return object
     */
    public function createTicket(array $ticketData)
    {
        $credentials = $this->getCredentials();
        try {
            $response = Http::withHeaders([
                'Content-Type' => 'application/json'
            ])->withOptions([
                'auth' => [$credentials->apiKey, $credentials->password]
            ])->post($credentials->baseUrl . 'tickets', [
                "description" => $this->formatComments($ticketData['description']),
                "subject" => $this->formatComments($ticketData['summary']),
                "email" => config('freshservice.email'),
                "priority" => config('freshservice.priority'),
                "status" => config('freshservice.status'),
            ])->throw();

            if ($response->successful()) {
                return true;
            }
        } catch (Exception $e) {
            logMessage('Freshservice action failed. ' . $e->getMessage(), 'error');
        }

        return true;
    }



    /**
     * get single ticket data using id
     *
     * @param  string $requestId
     * @return object
     */
    public function getTicket(string $requestId)
    {
        $credentials = $this->getCredentials();
        try {
            $response = Http::withHeaders([
                'Content-Type' => 'application/json'
            ])->withOptions([
                'auth' => [$credentials->apiKey, $credentials->password]
            ])->get($credentials->baseUrl . 'tickets/' . $requestId)
                ->throw();

            if ($response->successful()) {
                return $response->object();
            }
        } catch (Exception $e) {
            logMessage('Freshservice action failed. ' . $e->getMessage(), 'error');
        }

        return false;
    }

    /**
     * addCommentToTicket
     *
     * @param  string $requestId
     * @param  string $message
     * @param  string $userId
     * @return object
     */
    public function addCommentToTicket(string $requestId, string $message, string $userId = '')
    {
        $credentials = $this->getCredentials();
        try {
            Http::withHeaders([
                'Content-Type' => 'application/json'
            ])->withOptions([
                'auth' => [$credentials->apiKey, $credentials->password]
            ])->post($credentials->baseUrl . 'tickets/' . $requestId . '/reply', ["body" => $this->formatComments($message)])
                ->throw();
        } catch (Exception $e) {
            logMessage('Freshservice action failed. ' . $e->getMessage(), 'error');
            FailedTicketHistory::create([
                'user_id' => $userId,
                'ticket_id' => $requestId,
                'exceptions' => json_encode($e->getMessage()),
                'description' => $this->formatComments($message)
            ]);
        }
    }

    /**
     * check the API connection 
     */
    public function checkConnection()
    {
        $credentials = $this->getCredentials();
        try {
            $response = Http::withOptions([
                'auth' => [$credentials->apiKey, $credentials->password],
            ])->get($credentials->baseUrl . 'tickets/')
                ->throw();

            return $response->successful();
        } catch (Exception $e) {
            logMessage('Freshservice action failed: ' . $e->getMessage(), 'error');
            return false;
        }
    }

    /**
     * format comments before display remove unwanted
     */
    function formatComments($message)
    {
        $type = 'start';
        $pos = 0;

        foreach (str_split($message) as $char) {
            if ($char === '*') {
                if ($type == 'start') {
                    $message[$pos] = '~';
                } else {
                    $message[$pos] = '^';
                }

                $type = ($type == 'start') ? 'stop' : 'start';
            }
            $pos++;
        }



        $message = str_replace('\n\n', '<br><br>', $message);
        $message = str_replace('\n', '<br>', $message);
        $message = str_replace('~', '<strong>', $message);
        $message = str_replace('^', '</strong>', $message);

        return $message;
    }
    /**
     * Add a commet with attachment to ticket
     * @param string $ticketId
     * @param string $message
     * @param string $userId
     * @param mixed $filePath
     * @param mixed $fileName
     * 
     * @return [type]
     */
    public function addAttachmentToTicket(string $ticketId, string $message, string $userId, $filePath, $fileName)
    {
        $credentials = $this->getCredentials();
        $guzzle = $this->getClient($credentials->apiKey, 'test');
        try {
            $response = $guzzle->post('tickets/' . $ticketId . '/reply', [
                'headers' => ['X-Atlassian-Token' => 'no-check'],
                'multipart' => [
                    [
                        'name'     => 'attachments[]',
                        'contents' => fopen($filePath, 'r'),
                    ],
                    [
                        'name' => 'body',
                        'contents' => $message
                    ]
                ],


            ]);
            $response->withHeader('Content-Type', 'multipart/form-data');
            $response = $response->getBody()->getContents();
            $response = json_decode($response, true);
            return true;
        } catch (\Exception $e) {
            logMessage('Freshservice action failed. ' . $e->getMessage(), 'error');
            FailedTicketHistory::create([
                'user_id' => $userId,
                'ticket_id' => $ticketId,
                'exceptions' => json_encode($e->getMessage()),
                'description' => $this->formatComments($message)
            ]);
            return false;
            // throw new \ErrorException($e);
        }
    }

    /**
     * Get the complete ticket url
     * @param string $requestId
     * 
     * @return [type]
     */
    public function getTicketUrl($requestId)
    {
        // Cache the Freshservice URL for 60 minutes (adjust the time as needed).
        $freshServiceUrl = Cache::remember('freshservice_credential_url', 3600, function () {
            return optional(ApiCredential::where('slug', 'freshservice')->first())->url;
        });

        if ($freshServiceUrl) {
            // Example URL: https://domainservicedesk.freshservice.com/helpdesk/tickets/8
            return $requestId ? '<a href="' . $freshServiceUrl . 'helpdesk/tickets/' . $requestId . '" target="_blank">' . $requestId . '</a>' : '';
        }

        return '';
    }
}
