<?php

namespace App\Services\Integrations\SoftwareLicense;

use Exception;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class ZoomIntegration
{
    protected $name;
    protected $baseUrl;
    protected $apiVersion;
    protected $itemsPerpageSize = 50;

    /**
     * Constructor for ZoomIntegration
     * 
     */
    public function __construct()
    {
        $this->name = 'Zoom';
        $this->baseUrl = config('services.zoom.base_url');
        $this->apiVersion = config('services.zoom.api_version');
    }

    /**
     * It check software asset API connection status
     * 
     * @return int
     */
    public function checkConnection($accessToken)
    {
        $response = $this->getApiResponse($accessToken, $this->name . ' check connection', '/users/me');

        return  $response  ? true : false;
    }

    /**
     * It sends a POST request to the API endpoint with the data provided and returns the response in JSON format
     * 
     * @param accessToken
     * @param name The name of the API you are calling.
     * @param url The url of the API endpoint you want to call.
     * @param queryParams The data to be sent to the API.
     * 
     * @return mixed response from the API call.
     */
    public function getApiResponse($accessToken, $name, $url, $queryParams = [])
    {
        $apiURL =  $this->baseUrl . '/' . $this->apiVersion . $url;
        try {
            $response =  Http::withToken($accessToken)
                ->get($apiURL, $queryParams)
                ->throw();

            return $response ? $response->json() : '';
        } catch (Exception $e) {
            Log::channel('daily')->info($name . " fetch error ---- " . $e->getMessage());

            return false;
        }
    }

    /**
     * It generates the access token using authorization code received from softeware aseet api authentication
     *
     * @param code The authorization code you received from the previous step.
     * 
     * @return mixed response is a JSON object containing the access token, the token type, and the number of seconds until the token expires and refresh token.
     */
    public function generateAccessToken($userName, $key,  $password)
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => "Basic " . base64_encode($key . ':' . $password)
            ])
                ->post($this->baseUrl . '/oauth/token?grant_type=account_credentials&account_id=' . $userName)
                ->throw();
            $data = $response->json();

            return $data['access_token'] ?? '';
        } catch (Exception $e) {
            Log::channel('daily')->error('Zoom token generation : ' . $e->getMessage());

            return false;
        }
    }

    /**
     * It fetches team members of software asset from API
     * 
     * @param string $accessToken
     * @param mixed $nextPageToken
     * 
     * @return mixed
     */
    public function getSoftwareAssetMemberList($accessToken, $nextPageToken)
    {
        $url  = '/users?page_size=' . $this->itemsPerpageSize;

        if ($nextPageToken) {
            $url  =  '/users?page_size=' . $this->itemsPerpageSize . '&next_page_token=' . $nextPageToken;
        }

        return $this->getApiResponse($accessToken, $this->name . ' member list', $url);
    }

    /**
     * It fetches roles of the software asset from API
     * 
     * @param string $accessToken
     * 
     * @return array
     */
    public function getRolesList($accessToken)
    {
        $url  = '/roles';

        return $this->getApiResponse($accessToken, 'Fetch Zoom roles list', $url);
    }

    /**
     * This function updates the number of licenses purchased for a Dsoftware asset
     * 
     * @param string $accessToken
     * 
     * @return bool 
     */
    public function updateLicencePurchasedCount($accessToken)
    {
        $response  =  $this->getApiResponse($accessToken, $this->name . ' update license purchased count', '/users/summary');

        return $response['licensed_users_count'] ?? 0;
    }

    /**
     * Imports member hosted meetings from Zoom API.
     *
     * @param string $accessToken The access token for authentication.
     * @param array $zoomMember The Zoom member details.
     * @param string $fromDate The start date for importing meetings.
     * @param string $toDate The end date for importing meetings.
     * @param string $nextPageToken (optional) The token for the next page of results.
     * @return mixed The response from the API.
     */
    public function getMemberHostedMeetings($accessToken, $zoomMember, $fromDate, $toDate, $nextPageToken = '')
    {
        $response  =  $this->getApiResponse($accessToken, $this->name . ' get hosted meetings', '/users/' . $zoomMember['zoom_user_id'] . '/meetings',  ['type' => 'previous_meetings', 'from' => $fromDate, 'to' => $toDate, 'next_page_token' => $nextPageToken]);
        return $response;
    }
}
