<?php

namespace App\Services\Integrations\SoftwareLicense;

use Exception;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class DropboxIntegration
{
    protected $name;
    protected $baseUrl;
    protected $apiVersion;

    /**
     * Constructor for DropboxIntegration
     * 
     */
    public function __construct()
    {
        $this->name = 'Dropbox';
        $this->baseUrl = config('services.dropbox.base_url');
        $this->apiVersion = config('services.dropbox.api_version');
    }

    /**
     * It sends a POST request to the API endpoint with the data provided and returns the response in JSON format
     * 
     * @param name The name of the API you are calling.
     * @param url The url of the API endpoint you want to call.
     * @param data The data to be sent to the API.
     * 
     * @return mixed response from the API call.
     */
    public function getApiResponse($accessToken, $name, $url, $data='')
    {
        $apiURL =  $this->baseUrl . '/' . $this->apiVersion . $url;
        
        try {
            $response =  Http::withToken($accessToken)            
            ->retry(3, 5000)
            ->send('POST',$apiURL,[
                         'body' => $data ? json_encode($data) : json_encode(null)
                     ])
             ->json();          
            
             return $response;
        } catch   (Exception $e) {
            Log::channel('daily')->info($name." fetch error ---- " . $e->getMessage());
            
            return false;
        }
    }

    /**
     * It generates the access token using authorization code received from softeware aseet api authentication
     *
     * @param code The authorization code you received from the previous step.
     * 
     * @return mixed response is a JSON object containing the access token, the token type, and the number of seconds until the token expires and refresh token.
     */
    public function generateAccessToken($code, $redirectURL, $key,  $password)
    {
        try {
            $data = [
                'grant_type' => 'authorization_code',
                'code' => $code,
                'redirect_uri' => $redirectURL
            ];
            $response = Http::asForm()
            ->withBasicAuth($key,  $password)
            ->post($this->baseUrl . '/oauth2/token', $data)->throw()->json();

            return $response;
        } catch (Exception $e) {
            Log::channel('daily')->info($this->name . " Error :- " . $e->getMessage());

            return false;
        }
    }
    
    /**
     * It refreshes the token.
     * 
     * @return mixed response is a JSON object with the following fields:
     */
    public function refreshToken($tokenData, $key, $secret)
    {
        if (empty($tokenData)) {
            return false;
        }

        try {
            $refreshToken = $tokenData['refresh_token'] ?? '';
            $data = [
                'grant_type' => 'refresh_token',
                'refresh_token' => $refreshToken
            ];
            $response = Http::asForm()
            ->withBasicAuth($key, $secret)
            ->post($this->baseUrl . '/oauth2/token', $data)->throw()->json();

            return $response['access_token'] ?? false;
        } catch (Exception $e) {
            Log::channel('daily')->info($this->name . " refresh token Error :- " . $e->getMessage());

            return false;
        }
    }
    
    /**
     * It check software asset API connection status
     * 
     * @return int
     */
    public function checkConnection($accessToken)
    {
        $response = $this->getApiResponse($accessToken, $this->name . ' info', '/team/get_info');   
        
        return  $response  ? true : false;
    }
   
    /**
     * This function updates the number of licenses purchased for a Dsoftware asset
     * 
     * @param string $accessToken
     * 
     * @return bool 
     */
    public function updateLicencePurchasedCount($accessToken)
    {
        $response  =  $this->getApiResponse($accessToken, $this->name . ' info','/team/get_info');
        
        return $response['num_licensed_users'] ?? 0;
    }

    /**
     * It fetches team members of software asset from API
     * 
     * @param string $accessToken
     * @param mixed $cursor
     * @param bool $hasMore
     * 
     * @return mixed
     */
    public function getSoftwareAssetMemberList($accessToken, $cursor='', $hasMore=false)
    {
        $url  = '/team/members/list';
        $data = json_encode(null);

        if ($hasMore && $cursor!='') {
            $url  =  '/team/members/list/continue';
            $data = json_encode(['cursor' => $cursor]);
        }
        
        return $this->getApiResponse($accessToken, $this->name . ' member list', $url, $data);        
    }
}
