<?php

namespace App\Services\Integrations\Slack;

use App\Models\ApiCredential;
use Exception;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class SlackApiIntegration
{
	protected $credential;
	protected $client;

	public function __construct() {}

	/**
	 * Retrieves the credentials for the API.
	 *
	 * @return ApiCredential|null The credentials for the API, or null if not found.
	 */
	private function getTeqtivityAppCredentials()
	{
		return ApiCredential::where('slug', 'teqtivity_slack_notification_app')->first();
	}

	/**
	 * Retrieves the credentials for the API.
	 *
	 * @return ApiCredential|null The credentials for the API, or null if not found.
	 */
	private function getCredentials()
	{
		return ApiCredential::where('slug', 'slack_api')->first();
	}

	/**
	 * Retrieves an access token using the provided authorization code.
	 *
	 * @param string $authCode The authorization code to use for retrieving the access token.
	 * @return mixed Returns the access token if successful, otherwise false.
	 */
	public function getAccessToken($authCode)
	{
		if (!$authCode) {
			return false;
		}
		$credentials = $this->getTeqtivityAppCredentials();
		$response = Http::asForm()->post(config('services.slack.token_url'), [
			'client_id' => $credentials->user_name,
			'client_secret' => $credentials->password,
			'code' => $authCode,
			'grant_type' => 'authorization_code',
			'redirect_uri' => url('/slack-notifications-callback/'),
		]);

		// Check if the response is successful (HTTP status 200)
		if ($response->successful()) {
			$data = $response->json();
			return $data;
		} else {
			return false;
		}
	}

	/**
	 * Retrieves all channels.
	 *
	 * @param string $nextCursor The cursor for pagination.
	 * @return mixed Returns an array of channel data or false on failure.
	 */
	public function getAllChannels($nextCursor = '')
	{

		$url = config('services.slack.channel_list_url');

		$params = [
			'limit' => 200,
			'exclude_archived' => true,
			'cursor' => $nextCursor
		];

		return $this->callGetApi($url, $params);
	}

	/**
	 * Retrieves the information of a channel by its ID from the Slack API.
	 *
	 * @param string $channelId The ID of the channel. Default is an empty string.
	 * @return mixed|false The channel information as an associative array if successful, false otherwise.
	 */
	public function getChannelInfoById($channelId = '')
	{
		if (!$channelId === true) {
			return false;
		}

		$url = config('services.slack.channel_details_url');

		$params = [
			'channel' => $channelId
		];

		return $this->callGetApi($url, $params);
	}

	/**
	 * Calls the GET API with the given URL and parameters.
	 *
	 * @param string $url The URL of the API endpoint.
	 * @param array $params The parameters to be sent with the request.
	 * @return mixed|false The response data if successful, false otherwise.
	 */
	private function callGetApi($url, $params)
	{

		$credentials = $this->getCredentials();
		$accessToken = $credentials->key;

		$response = Http::withToken($accessToken)->get($url, $params);

		if ($response->successful()) {
			$data = $response->json();

			if ($data['ok'] == 'true') {
				return $data;
			}

			return false;
		} else {
			return false;
		}
	}

	/**
	 * Adds an app to a Slack channel.
	 *
	 * @param int $channelId The ID of the channel to add the app to.
	 * @throws Exception If an error occurs while adding the app to the channel.
	 * @return bool Returns true if the app was successfully added to the channel, false otherwise.
	 */
	public function addAppToChannel($channelId)
	{
		$credentials = $this->getCredentials();
		$accessToken = $credentials->password; // the password has users token

		try {
			$response = Http::withToken($accessToken)->post(config('services.slack.add_to_conversation_url'), [
				'channel' => $channelId,
				'users' => [$credentials->user_name],
			]);

			if ($response->successful() && $response->json()['ok'] == 'true') {
				return true;
			}
			Log::channel('daily')->error($response->json()['error'] ?? 'error');
			return false;
		} catch (Exception $e) {
			Log::channel('daily')->error($e->getMessage() . ' ' . $e->getFile() . ' ' . $e->getLine());
			return false;
		}
	}

	/**
	 * Sends a message to a Slack channel.
	 *
	 * @param int $channelId The ID of the channel.
	 * @param string $message The message to send.
	 * @param  $attachments The attachments to send , can be interactive buttons.
	 * @throws Exception If an error occurs while sending the message.
	 * @return bool Returns `true` if the message was sent successfully, `false` otherwise.
	 */
	public function sendMessageToChannel($channelId, $message, $attachments = null)
	{
		$credentials = $this->getCredentials();
		$accessToken = $credentials->key;

		$postData = [
			'channel' => $channelId,
			'text' => $message
		];

		if (!empty($attachments)) {
			$postData['attachments'] = json_encode($attachments['attachments']);
		}

		try {
			$response = Http::withToken($accessToken)->post(config('services.slack.post_message_url'), $postData);

			if ($response->successful() && $response->json()['ok'] == 'true') {
				return true;
			}

			return false;
		} catch (Exception $e) {
			Log::channel('daily')->error($e->getMessage() . ' ' . $e->getFile() . ' ' . $e->getLine());
			return false;
		}
	}

	/**
	 * Retrieve a single user by looking them up by their registered email address
	 *
	 * @param $email   search email
	 *
	 * @return Array
	 */
	public function searchUser($email)
	{
		try {

			$credentials = $this->getCredentials();
			$accessToken = $credentials->key;

			$response = Http::withToken($accessToken)->get(config('services.slack.search_user_api_url'), [
				'email' => $email,
			]);

			if ($response->successful()) {
				$data = $response->json();

				if ($data['ok'] == 'true') {
					return $data;
				}
				return false;
			} else {
				return false;
			}
		} catch (\Exception $e) {
			return ['ok' => false, 'error' => true];
		}
	}

	/**
	 * Update a message on Slack.
	 *
	 * @param mixed $channelId The ID of the channel.
	 * @param string $message The message to update.
	 * @param array|null $attachments The attachments to update, if any.
	 * @param mixed|null $messageTs The timestamp of the message to update, if applicable.
	 * @throws Exception If an error occurs while updating the message.
	 * @return bool Returns true if the message was updated successfully, false otherwise.
	 */
	public function updateSlackMessage($channelId, $message, $attachments = null, $messageTs = null)
	{
		$credentials = $this->getCredentials();
		$accessToken = $credentials->key;

		$postData = [
			'channel' => $channelId,
			'text' => $message,
			'ts' => $messageTs,
		];

		if (!empty($attachments)) {
			$postData['attachments'] = json_encode($attachments['attachments']);
		}

		try {
			$response = Http::withToken($accessToken)->post(config('services.slack.update_message_url'), $postData);

			if ($response->successful() && $response->json()['ok'] == 'true') {
				return true;
			}

			return false;
		} catch (Exception $e) {
			Log::channel('daily')->error($e->getMessage() . ' ' . $e->getFile() . ' ' . $e->getLine());
			return false;
		}
	}
}
