<?php

namespace App\Services\Integrations\ShipmentTracking;

use App\Models\ApiCredential;
use App\Services\Integrations\ShipmentTracking\UPS\Ups;
use GuzzleHttp\Client;
use Exception;
use Illuminate\Support\Facades\Log;


/** 
 * Service class for tracking the shipment status by using UPS Api
 */
class UpsTracking implements ShipmentTrackingInterface
{
    private $apiCredential;
    private $client;
    private $serviceClass;
    public function __construct(Ups $serviceClass)
    {
        $this->client             = new Client();
        $this->apiCredential      = ApiCredential::where('slug', 'ups_tracking')->first();
        $this->serviceClass       = $serviceClass;
    }

    /**
     * Get the latest tracking details by tracking number
     *
     * @param $tracking_no   UPS shipment tracking no
     * @return String
     */
    public function getTrackingDetails($trackingNo)
    {
        $shipmentData   = $this->shipmentTrackingDetails($trackingNo);
        $shipStatus     = $this->shipmentLatestStatus($shipmentData);

        return $shipStatus;
    }

    /**
     * Fetch the tracking details of the UPS shipment by tracking number
     *
     * @param $client        GuzzleHttp\Client object
     * @param $tracking_no   UPS shipment tracking no
     * @return array UPS API response
     */
    public function shipmentTrackingDetails($tracking_no)
    {
        if (!$this->apiCredential) {
            return false;
        }
        $response = $this->client->post($this->apiCredential->url, [
            \GuzzleHttp\RequestOptions::JSON => [
                'UPSSecurity'  => [
                    'UsernameToken'     => [
                        "Username" => $this->apiCredential->user_name,
                        "Password" => $this->apiCredential->password,
                    ],
                    'ServiceAccessToken' => [
                        "AccessLicenseNumber" => $this->apiCredential->key,
                    ],
                ],
                'TrackRequest' => [
                    'InquiryNumber' => $tracking_no,
                ],
            ]
        ]);
        return json_decode($response->getBody()->getContents());
    }

    /**
     * Taking the latest shipment status from the UPS api response data
     *
     * @param $shipmentData   UPS shipment API response data
     * @return String 
     */
    public function shipmentLatestStatus($shipmentData)
    {
        $shipmentStatus = array();
        if (isset($shipmentData->TrackResponse)) { //Success response
            $shipmentStatus['status']  = $shipmentData->TrackResponse->Shipment ?
                ($shipmentData->TrackResponse->Shipment->Package ?
                    (is_array($shipmentData->TrackResponse->Shipment->Package) ?
                        $this->latestStatusCode($shipmentData->TrackResponse->Shipment->Package[0])
                        :
                        $this->latestStatusCode($shipmentData->TrackResponse->Shipment->Package)
                    )
                    : null)
                : null;

            $shipmentStatus['updated_on']  = $shipmentData->TrackResponse->Shipment ?
                ($shipmentData->TrackResponse->Shipment->Package ?
                    (is_array($shipmentData->TrackResponse->Shipment->Package) ?
                        $this->updatedOn($shipmentData->TrackResponse->Shipment->Package[0])
                        :
                        $this->updatedOn($shipmentData->TrackResponse->Shipment->Package)
                    )
                    : null)
                : null;
        } else if (isset($shipmentData->Fault)) { //Error response
            $shipmentStatus['updated_on']  = null;
            $shipmentStatus['status']  = $shipmentData->Fault->detail ?
                ($shipmentData->Fault->detail->Errors ?
                    ($shipmentData->Fault->detail->Errors->ErrorDetail ?
                        ($shipmentData->Fault->detail->Errors->ErrorDetail->PrimaryErrorCode ?
                            $shipmentData->Fault->detail->Errors->ErrorDetail->PrimaryErrorCode->Description
                            : null)
                        : null)
                    : null)
                : null;
        }
        return $shipmentStatus;
    }

    /** 
     * Taking the latest ups shipment status from package enitity
     */
    public function latestStatusCode($package)
    {
        return ($package->Activity ?
            ($package->Activity->Status ?
                $package->Activity->Status->Description
                : null)
            : null);
    }

    /** 
     * Taking the latest ups shipment status update date from package enitity
     */
    public function updatedOn($package)
    {
        $date = ($package->Activity ?
            $package->Activity->Date
            : null);

        if (!$date) {
            return null;
        }

        return date("Y-m-d", strtotime($date));
    }

    /**
     * Get the latest tracking details by tracking number
     *
     * @param $tracking_no   UPS shipment tracking no
     * @return String
     */
    public function getLatestTrackingDetails($trackingNo)
    {
        $shipmentData   = $this->shipmentTrackingDetails($trackingNo);
        $shipStatus     = $this->serviceClass->getShipmentLatestStatusName($shipmentData);

        return $shipStatus;
    }
}
