<?php 

namespace App\Services\Integrations\ShipmentTracking\FedEx;

/** 
 * Service class for Fedex api implimentation
 */

class FedEx 
{

	private $soapClient;
	private $request;
	private $endPoint;	
	private $key;
	private $password;
	private $accountNo;
	private $meterNo;
	private $serviceId;
	private $major;
	private $intermediate;
	private $minor;
	private $customerTransactionId;

	/**
	 * 	Create a new FedEx instance. Requires a WSDL file, and FedEx API credentials.
	 *
	 *	@param string 	// WSDL File
	 *	@param string 	// Key
	 * 	@param string 	// Password
	 *	@param string 	// Account Number
	 *	@param string 	// Meter number
	 * 	@return void
	 */
	public function __construct($wsdlFile, $key, $password, $accountNo, $meterNo)
	{
		$wsdlPath = __DIR__.'/_wsdl/'.$wsdlFile;
		// Will throw a SoapFault exception if wsdlPath is unreachable
		$this->soapClient = new \SoapClient($wsdlPath, array('trace' => true));	      

		// Initialize request to an empty array
		$this->request = array();

		$this->customerTransactionId = "Defualt ID";

		$this->key 		 = $key;
		$this->password  = $password;
		$this->accountNo = $accountNo;
		$this->meterNo 	 = $meterNo;		
	}

	/**
	 * 	Set SoapClient ednpoint.
	 *
	 *	@param SoapClient endpoint
	 *  @return Null
	 */
	public function setEndpoint($endPoint)
    {
    	$newLocation = $this->soapClient->__setLocation($endPoint);
    }

	/**
	 * 	Retrieve the SoapClient object.
	 *
	 *	@return SoapClient
	 */
	public function getSoapClient()
    {
        return $this->soapClient;
    }

    /**
     *	Sets the TransactionDetail/CustomerTransactionId for the current request.
     *
     *	@param string 	// Any text up to 40 characters
     *	@return void
     */
    public function setCustomerTransactionId($id)
    {
    	$this->customerTransactionId = $id;
    }

    /**
     *	Sets the API service to be used and the version information.
     *
     *	@param string 	// Service to use, ie. "trck"
     * 	@param int 		// Major version to use, ie. 9
     * 	@param int 		// Intermediate version to use, ie. 1
     * 	@param int 		// Minor version to use, ie. 0
     */
    public function setVersion($serviceId, $major, $intermediate, $minor)
    {
    	$this->serviceId = $serviceId;
    	$this->major = $major;
    	$this->intermediate = $intermediate;
    	$this->minor = $minor;
    }

    /**
     * 	Builds and returns the basic request array to be sent with each request to the FedEx API. This assumes the
     *	setVersion method has already been called by an extended class.
   	 *
     *	Takes an optional parameter, $addReq. This parameter is used to set the additonal request details. These details
     *	are determined by the particular service being called and are passed by the extended service classes.
     *
     *	@param array 	// Additonal request details
     */
    public function buildRequest($addReq = [])
    {
    	// Build Authentication
    	$this->request['WebAuthenticationDetail'] = array(
			'UserCredential'=> array(
				'Key' 		=> $this->key, 
				'Password'	=> $this->password
			)
		);

		//Build Client Detail
		$this->request['ClientDetail'] = array(
			'AccountNumber' => $this->accountNo, 
			'MeterNumber' 	=> $this->meterNo
		);

		// Build Customer Transaction Id
		$this->request['TransactionDetail'] = array(
			'CustomerTransactionId' => $this->customerTransactionId
		);
		
		// Build API Version info
		$this->request['Version'] = array(
			'ServiceId' 	=> $this->serviceId, 
			'Major' 		=> $this->major, 
			'Intermediate'	=> $this->intermediate, 
			'Minor' 		=> $this->minor
		);

		// Enable detailed scans
		//$this->request['ProcessingOptions'] = 'INCLUDE_DETAILED_SCANS';

		$this->request = array_merge($this->request, $addReq);

		return $this->request;
    }

    /**
     * Gets the status description from fedex api respose TrackDetails object
     *
     * @param $trackingData->CompletedTrackDetails->TrackDetails Object
     * @return Array
     */
    public function getStatusDescription($trackDetails)
    {
    	$response['status']     = null;
    	$response['updated_on'] = null;
    	
    	if ($trackDetails->Notification->Severity == 'ERROR') {
    		$response['status'] = 'ERROR';
    		return $response;
    	}

        $response['status'] 	= $this->getTrackingStatus($trackDetails);
        $response['updated_on'] = $this->getTrackingStatusUpdatedDate($trackDetails);

        return $response;
    }

    /**
     * Get the Latest shipment status from the $trackDetails Object
     *
     * @param $trackingData->CompletedTrackDetails->TrackDetails Object
     * @return String
     */
    public function getTrackingStatus($trackDetails)
    {
    	return isset($trackDetails->StatusDetail) ?
                    (isset($trackDetails->StatusDetail->Description) ?
                        $trackDetails->StatusDetail->Description
                    : null)
                : null;
    }

    /**
     * Get the Latest shipment status updated date from the $trackDetails Object
     *
     * @param $trackingData->CompletedTrackDetails->TrackDetails Object
     * @return Date string
     */
    public function getTrackingStatusUpdatedDate($trackDetails)
    {
    	return isset($trackDetails->StatusDetail) ?
                    (isset($trackDetails->StatusDetail->CreationTime) ?
                        format_fedex_date($trackDetails->StatusDetail->CreationTime)
                    : null)
                : null;
    }


    /**
     * If there is have multiple packages, then taking the fist package `TrackingNumberUniqueIdentifier` value
     *
     * @param $trackingData   FedEx API response array
     * @return Array 
     */
    public function getTrackingNumberUniqueIdentifier($trackingData)
    {
        if ($trackingData->HighestSeverity == 'SUCCESS') {
            return $trackingData->CompletedTrackDetails ?
                        ($trackingData->CompletedTrackDetails->TrackDetails ?
                            (is_array($trackingData->CompletedTrackDetails->TrackDetails) ?
                                $trackingData->CompletedTrackDetails->TrackDetails[0]->TrackingNumberUniqueIdentifier
                            :
                                null
                            )
                        : null)
                    : null;
        }
        return null;
    }


    /**
     * Taking the latest shipment status name from the FedEx api response data
     *
     * @param $trackingData   FedEx API response array
     * @return Array 
     */
    public function getShipmentLatestStatusName($trackingData)
    {
        $response['status']     = null;
        $response['updated_on'] = null;
        
        if ($trackingData->HighestSeverity == 'SUCCESS') {
            return $trackingData->CompletedTrackDetails ?
                        ($trackingData->CompletedTrackDetails->TrackDetails ?
                            (is_array($trackingData->CompletedTrackDetails->TrackDetails) ?
                                $response
                            :
                                $this->getStatusDescription($trackingData->CompletedTrackDetails->TrackDetails)
                            )
                        : $response)
                    : $response;
        }
        return $response;
    }
}