<?php

namespace App\Services\Integrations\Security\Crowdstrike;

use Exception;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class CrowdstrikeIntegration
{
    protected $name;
    protected $baseUrl;
    protected $itemsPerpageSize = 200;

    /**
     * Constructor for CrowdstrikeIntegration
     * 
     */
    public function __construct()
    {
        $this->name = 'Crowdstrike';
        $this->baseUrl = config('services.crowdstrike.base_url');
    }

    /**
     * It generates the access token using authorization code received from Crowdstrike api authentication
     *
     * @param string $key client ID
     * @param string $password client secret
     * 
     * @return mixed response is a JSON object containing the access token, the token type, and the number of seconds until the token expires and refresh token.
     */
    public function generateAccessToken($key, $password)
    {
        try {
            $response = Http::asForm()
                ->withHeaders(['user_agent' => config('services.crowdstrike.user_agent')])
                ->post($this->baseUrl . '/oauth2/token', [
                    'client_id' => $key,
                    'client_secret' => $password
                ])->throw();

            $data = $response->json();

            return $data ?? '';
        } catch (Exception $e) {
            Log::channel('daily')->error($this->name . ' token generation : ' . $e->getMessage());

            return false;
        }
    }

    /**
     * It sends a POST request to the API endpoint with the data provided and returns the response in JSON format
     * 
     * @param accessToken
     * @param name The name of the API you are calling.
     * @param url The url of the API endpoint you want to call.
     * @param data The data to be sent to the API.
     * 
     * @return mixed response from the API call.
     */
    public function getApiResponse($accessToken, $name, $url, $data = [])
    {

        $apiURL =  $this->baseUrl . $url;
        $params = $data;

        try {
            $response =  Http::withToken($accessToken)
                ->withHeaders(['user_agent' => config('services.crowdstrike.user_agent')])
                ->get($apiURL, $params)
                ->throw();

            return $response ? $response->json() : '';
        } catch (Exception $e) {
            Log::channel('daily')->info($name . " fetch error ---- " . $e->getMessage());

            return false;
        }
    }

    /**
     * Retrieves the device IDs based on the access token, offset, and limit.
     *
     * @param string $accessToken The access token used for authentication.
     * @param int $offset The offset value for pagination.
     * @throws Some_Exception_Class A description of the exception that can be thrown.
     * @return array The array of device IDs.
     */
    public function getDeviceIds($accessToken, $offset)
    {
        $url = '/devices/queries/devices/v1';
        $data = [
            'offset' => $offset,
            'limit' => $this->itemsPerpageSize,
        ];
        $response = $this->getApiResponse($accessToken, $this->name . 'get device ids', $url, $data);

        return $response ?? [];
    }

    /**
     * Retrieves device data using the provided access token and device ID.
     *
     * @param string $accessToken The access token used for authentication.
     * @param string $deviceId    The ID of the device to retrieve data for.
     *
     * @throws Some_Exception_Class A description of the exception that can be thrown.
     *
     * @return array  The retrieved device data.
     */
    public function getDeviceData($accessToken, $deviceId)
    {
        $url = '/devices/entities/devices/v1';
        $data = [
            'ids' => $deviceId
        ];
        $response = $this->getApiResponse($accessToken, $this->name . 'get device data', $url, $data);

        return $response ?? [];
    }

    /**
     * Retrieves vulnerability data for a given device.
     *
     * @param string $accessToken The access token for authentication.
     * @param string $filterString The filter string.
     * @param string $paginationAfterToken The pagination token to get the next page of results.
     *
     * @throws Some_Exception_Class description of exception
     * @return array The vulnerability data for the device.
     */
    public function getVulnerabilityData($accessToken, $filterString, $paginationAfterToken)
    {
        $url = '/spotlight/combined/vulnerabilities/v1';
        $data = [
            'filter' => $filterString,
            'facet' => 'cve',
            'sort'  => 'created_timestamp.desc',
            'limit' => $this->itemsPerpageSize,
            'after' => $paginationAfterToken
        ];
        $response = $this->getApiResponse($accessToken, $this->name . 'get device vulnerability data', $url, $data);


        return $response ?? [];
    }
}
