<?php

namespace App\Services\Integrations\Retriever;

use App\Models\ApiCredential;
use Exception;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class RetrieverIntegrationService
{

    /**
     * Constructor for RetrieverIntegrationService class.
     *
     * @param RetrieverDataSyncService $syncService The data synchronization service for Retriever.
     */
    public function __construct()
    {
    }

    /**
     * Get the API credentials
     */
    public function getApiCredentials()
    {
        return ApiCredential::where('slug', 'retriever')->first();
    }

    /**
     * Check whether the API is connected or not
     */
    public function checkApiConnection()
    {
        $credential = $this->getApiCredentials();
        try {
            $key = optional($credential)->key;
            $response = Http::withToken($key)
                ->retry(2, 1000)
                ->get($credential->url . 'api/v1/device_returns/?cursor=')
                ->throw();

            return $response->successful();
        } catch (Exception $e) {
            Log::channel('single')->info("Retriever API connection error :- " . $e->getMessage());
            return false;
        }
    }

    /**
     * Sync the Return Device Details
     */
    public function getRetrieverDevices($nextPageLink = '')
    {
        $credential = $this->getApiCredentials();

        if ($nextPageLink === '') {
            $nextPageLink = $credential->url . 'api/v1/device_returns/';
        }

        try {
            $key = optional($credential)->key;
            $response = Http::withToken($key)
                ->retry(2, 1000)
                ->get($nextPageLink)
                ->throw()
                ->object();

            return $response;
        } catch (Exception $e) {
            Log::channel('single')->info("Retriever API connection error :- " . $e->getMessage());

            return false;
        }
    }

    /**
     * Create a shipment request to the Retriever API.
     *
     * @param array $shipmentData The data for the shipment request.
     * @throws Exception If there is an error creating the shipment request.
     * @return mixed|false The response from the API or false if an error occurred.
     */
    public function CreateShipmentRequest($shipmentData)
    {
        try {
            $credential = $this->getApiCredentials();

            $response = Http::withToken(optional($credential)->key)
                ->post($credential->url . 'api/v2/device_returns/', $shipmentData);

            return $response;
        } catch (Exception $e) {
            // dd($e->getMessage());
            Log::channel('daily')->error("Retriever create shipment error :- " . json_encode($e->getMessage()) . '\n' . json_encode($shipmentData));

            return false;
        }
    }
}
