<?php

namespace App\Services\Integrations;

use App\Events\BulkUpdates;
use App\Models\ApiCredential;
use Facades\App\Repositories\OktaDirectoryRepository;
use App\User as UserModel;
use App\Models\UserType;
use Exception;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class OktaDirectoryIntegration implements UsersDirectoryIntegration
{
    /**
     * Returns the OktaDirectory credentials
     * @return [type]
     */
    public function getOktaCredentials()
    {
        return ApiCredential::where('slug', 'okta_directory')->first();
    }

    /**
     * Call okta api and fetch updated users between dates
     *
     * @return array
     */
    public function importOktaDirectoryData($dateFrom, $dateTo)
    {
        $dateFrom = $dateFrom->format('Y-m-d\TH:i:s.v\Z');
        $dateTo = $dateTo->format('Y-m-d\TH:i:s.v\Z');
        $count = 0;
        try {
            $response = $this->getOktaUsersApi($dateFrom, $dateTo);

            if ($response->status() !== 200) {
                return null;
            }

            $result = $response->object();
            $count += $this->saveUser($result);
            $nextLink = $this->getNextPageLink($response->headers());
            while (!empty($nextLink)) {
                $response = $this->getOktaUsersApi($dateFrom, $dateTo, $nextLink);
                $result = $response->object();
                $count += $this->saveUser($result);
                $nextLink = $this->getNextPageLink($response->headers());
            }
            return $count;
        } catch (Exception $e) {
            return null;
        }
    }

    /**
     * API for getting the okta directory users 
     * @param mixed $dateFrom
     * @param mixed $dateTo
     * @param mixed $dateTo
     * 
     * @return [type]
     */
    public function getOktaUsersApi($dateFrom, $dateTo, $nextUrl = '')
    {
        $credential = $this->getOktaCredentials();
        $filter = '?filter=lastUpdated gt "' . $dateFrom . '" and lastUpdated lt "' . $dateTo . '"';
        // $filter = '?limit=20';
        $url = $credential->url . 'api/v1/users' . $filter;
        if ($nextUrl != '') {
            $url = $nextUrl;
        }
        // echo $url . PHP_EOL;
        try {
            $response = Http::withHeaders([
                'Content-Type' => 'application/json',
                'Authorization' => 'SSWS ' . $credential->key
            ])->get($url);
            return $response;
        } catch (\Exception $e) {
            // dd($e->getMessage());
            return null;
        }
        return false;
    }


    /**
     * Returns the next page link, here the next page link is present in the api responseHeaders
     * @param mixed $responseHeaders
     * 
     * @return [type]
     */
    private function getNextPageLink($responseHeaders)
    {
        $nextLink = null;
        if (isset($responseHeaders['link'])) {
            $links = $responseHeaders['link'];
            foreach ($links as $link) {
                // In the links array the rel="next"' indicated next page link
                if (strpos($link, 'rel="next"') !== false) {
                    // Gets the next page url from the links array the link is conatined inside < > characters
                    $pattern = '/<([^>]+)>/';
                    if (preg_match($pattern, $link, $matches)) {
                        $nextLink = $matches[1];
                    } else {
                        $nextLink = '';
                    }
                    break;
                }
            }
        }
        return $nextLink;
    }

    /**
     * Update the users table
     *
     * @return integer
     */
    public function saveUser($response)
    {
        $count = 0;
        $userTypeId = UserType::people()->first()->id;
        foreach ($response as $key => $item) {
            $userData = $this->generateUserData($item);
            $user = UserModel::where('okta_id', $item->id)->orWhere('email', optional($item->profile)->email)->first();
            if (!$user) {
                $userData['user_type_id'] = $userTypeId;
            }
            try {
                UserModel::updateOrCreate(['email' => $userData['email']], $userData);
                $count++;
            } catch (Exception $e) {
                // dd($e->getMessage());
                Log::channel('single')->info($userData['email'] . " Cannot be added");
            }
            if ($count % 100 == 1) {
                sleep(1);
                Log::channel('single')->info($count . "Users Updated");
            }
        }
        /*$assetHistory = [
            'action' => 'user_added',
            'description' => 'Users Added',
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'created_by' => 'Okta Directory'
        ];
        event(new BulkUpdates($assetHistory));*/
        return $count;
    }


    /**
     * Map the api response data with the table columns
     *
     * @return array
     */
    public function generateUserData($item)
    {
        $headerMap = Config('okta-directory.oktaDirectoryUserData');
        if (empty($item)) {
            return false;
        }
        foreach ($headerMap as $dbField => $fileHeader) {
            $profile = $item->profile;
            $result = $this->getRelationalValues($profile, $dbField, $fileHeader);
            if ($result !== false) {
                $profile->$fileHeader = $result;
            }

            $userData[$dbField] = $profile->$fileHeader ?? null;
        }
        $userData['okta_id'] = $item->id;
        $userData['status'] = $this->getStatusValues($item);
        $userData['updated_at'] = $this->getUpdatedAtDate($item);
        $userData['created_at'] = $this->getCreatedAtDate($item);
        $userData['terminated_date'] = $this->getTerminationDate($item, $userData);

        return $userData;
    }

    /**
     * Maps the api response data to the teqtivity ids
     * @param mixed $item
     * @param mixed $dbField
     * @param mixed $fileHeader
     * 
     * @return [type]
     */
    public function getRelationalValues($item, $dbField, $fileHeader)
    {
        // $item->originalhiredate = '2021-12-10';
        if ($dbField == "position_id") {
            return OktaDirectoryRepository::getPositionId($item, $dbField, $fileHeader);
        }

        if ($dbField == "department_id") {
            return OktaDirectoryRepository::getDepartmentId($item, $dbField, $fileHeader);
        }

        if ($dbField == 'employee_type_id') {
            return OktaDirectoryRepository::getEmployeeTypeId($item, $dbField, $fileHeader);
        }

        if ($dbField == 'status') {
            return OktaDirectoryRepository::getStatus($item, $dbField, $fileHeader);
        }

        if ($dbField == 'country_id') {
            return OktaDirectoryRepository::getCountryId($item, $dbField, $fileHeader);
        }

        if ($dbField == 'original_hire_date') {
            if (isset($item->$fileHeader)) {
                return parse_date_okta_hire_date($item->$fileHeader) ? parse_date_okta_hire_date($item->$fileHeader) : null;
            }

            return null;
        }

        if ($dbField == 'hire_date') {
            if (isset($item->$fileHeader)) {
                return parse_date_okta_hire_date($item->$fileHeader) ? parse_date_okta_hire_date($item->$fileHeader) : null;
            }

            return null;
        }

        if ($dbField == 'manager_id') {
            return OktaDirectoryRepository::getManagerId($item, $dbField, $fileHeader);
        }

        if ($dbField == 'division_id') {
            return OktaDirectoryRepository::getDivisionId($item, $dbField, $fileHeader);
        }

        return false;
    }

    /**
     * Gets user status value
     * @param mixed $item
     * 
     * @return [type]
     */
    public function getStatusValues($item)
    {
        return OktaDirectoryRepository::getStatus($item);
    }


    /**
     * Gets the last updated date
     * @param mixed $item
     * 
     * @return [type]
     */
    public function getUpdatedAtDate($item)
    {
        //return $item->lastUpdated;
        return OktaDirectoryRepository::getUpdatedAtDate($item);
    }

    /**
     * Gets the created date
     * @param mixed $item
     * 
     * @return [type]
     */
    public function getCreatedAtDate($item)
    {
        //return $item->lastUpdated;
        return OktaDirectoryRepository::getCreatedAtDate($item);
    }

    /**
     * Taking the user termination date from the Okta status change Date
     */
    public function getTerminationDate($item, $userData)
    {
        if ($userData['status'] == 0) {
            //return $item->statusChanged;
            return OktaDirectoryRepository::getStatusChangedDate($item);
        }

        return null;
    }
}
