<?php

namespace App\Services\Integrations;

use App\Http\Traits\AutoCreateAssignTrait;
use App\Models\Asset;
use App\Models\ApiCredential;
use App\Models\AssetStatus;
use App\Models\AssetType;
use Illuminate\Support\Facades\Http;
use GuzzleHttp\Client;
use Exception;
use Illuminate\Support\Facades\Log;
use Illuminate\Http\Client\RequestException;
use GuzzleHttp\Psr7;
use App\Models\MobileIron;
use App\Repositories\DiscoveryTools\MobileIronRepository;
use Cache;

/**
 * Service Class for MobileIron API
 */
class MobileIronApi
{
    use AutoCreateAssignTrait;

    protected $api;
    protected $baseUrl;
    protected $userName;
    protected $password;

    public function __construct(protected MobileIronRepository $repository)
    {
        $this->api = ApiCredential::where('slug', 'mobile_iron')->first();
        $this->baseUrl  = optional($this->api)->url;
        $this->userName = optional($this->api)->user_name;
        $this->password = optional($this->api)->password;
    }

    /**
     * Upload the MobileIron API date to Database
     */
    public function upload()
    {
        $page = 0;
        while (1 == 1) {
            $response   = $this->getApiData($page);
            if ($response) {
                $response   = $this->saveMobileIronData($response);
            } else {
                break;
            }
            Log::channel('single')->info("MobileIron Importing Page :- " . $page . '-');
            $page++;
        }

        return true;
    }

    /**
     * Call the API and return the reponse
     *
     * @param int $page  page number
     * @return Array
     */
    public function getApiData($page = 0)
    {
        $dmPartitionId = $this->getDmPartitionId();
        $limit  = 50;
        $offset = $page * $limit;

        try {
            $response = Http::withBasicAuth($this->userName, $this->password)
                ->retry(3, 3000)
                ->get($this->baseUrl . "api/v1/device?q=&rows=$limit&start=$offset&sortField=id&sortFieldOrder=ASC&dmPartitionId=" . $dmPartitionId)
                ->throw()
                ->json();
            if ($response && isset($response['result'])) {
                return $response['result']['searchResults'];
            }
        } catch (RequestException $e) {
            Log::channel('single')->info("MobileIron Error Error :- " . $e->getMessage());
        }

        return false;
    }

    /**
     * Get the default device space id
     */

    public function getDmPartitionId()
    {
        try {
            $response = Http::withBasicAuth($this->userName, $this->password)->retry(3, 1000)->get($this->baseUrl . "api/v1/account?metadata")->throw()->json();

            if ($response && isset($response['result'])) {
                return $response['result']['defaultDmPartitionId'] ?? false;
            }
        } catch (RequestException $e) {
            Log::channel('single')->info("MobileIron Error :- " . $e->getMessage());
        }

        return false;
    }

    /**
     * Check API Connection status
     */
    public function checkApiConnection()
    {
        try {
            $response = Http::withBasicAuth($this->userName, $this->password)
                ->retry(3, 1000)
                ->get($this->baseUrl . "api/v1/account?metadata")
                ->throw()
                ->json();

            if ($response && isset($response['result'])) {
                return true;
            }
        } catch (RequestException $e) {
            Log::channel('single')->info("MobileIron Error :- " . $e->getMessage());
        }

        return false;
    }


    /**
     * Save the MobileIron data to DB.
     *
     * @param Array $response  APi response
     *
     * Return boolean
     */

    public function saveMobileIronData($response)
    {
        $dataToUpdateArray = [];

        foreach ($response as $key => $data) {
            $dataToUpdateMobileiron = $this->parseMobileIronResponse($data);
            $data['user_id']   = $this->getUserId($data['emailAddress']);
            $data['serial_no'] = $dataToUpdateMobileiron['serial_no'];
            $data['totalCapacity'] = $dataToUpdateMobileiron['totalCapacity'];

            $mobileIronData = MobileIron::updateOrCreate(
                ['serial_no' => $dataToUpdateMobileiron['serial_no']],
                $dataToUpdateMobileiron
            );

            $assetTypeId   = AssetType::firstOrCreate(['name' => 'Computer', 'slug' => 'computer'])->id;
            $assetStatusId = AssetStatus::firstOrCreate(['name' => 'Assigned', 'slug' => 'assigned'])->id;
            $dataToUpdate['user_id'] = $this->getUserId($data['user']['email'] ?? null);
            $dataToUpdate['asset_type_id'] = $assetTypeId;

            //Auto create and assign asset if asset is not exist
            if ($dataToUpdateMobileiron['asset_id'] == 0 && $dataToUpdateMobileiron['user_id'] != 0) {
                $techSpec = $this->getTechSpec($dataToUpdateMobileiron, 'mobile_iron_model');
                $dataToUpdateMobileiron['asset_status_id'] = $assetStatusId;
                $dataToUpdateMobileiron['mobile_iron_id']       = $mobileIronData->id;

                $this->autoCreateAndAssign($dataToUpdateMobileiron, $techSpec, "MobileIron");
            }

            if ($key == 500) {
                sleep(2);
            }
        }

        return true;
    }

    /**
     * Parse the code42 respone and create an arrray for insert into DB
     *
     * @param $response  MobileIron api response
     *
     * @return Array
     */
    public function parseMobileIronResponse($response)
    {
        if ($response) {
            $assetId    = $this->getAssetId($response['wifiMacAddress']); // $response['serialNumber']
            $userId     = $this->getUserId($response['emailAddress']);
            $carrierId  = $this->getCarrierId($response['currentCarrierNetwork']);

            $data['asset_id']       = $assetId;
            $data['user_id']        = $userId;
            // $data['serial_no']      = $response['serialNumber'] ?? '';
            $data['serial_no']      = $response['wifiMacAddress'] ?? '';
            $data['first_name']     = $response['firstName'] ?? '';
            $data['last_name']      = $response['lastName'] ?? '';
            $data['email']          = $response['emailAddress'] ?? '';
            $data['manufacturer']   = $response['manufacturer'] ?? '';
            $data['totalCapacity']  = convertSpecsToReadableValue($response['totalCapacity'] ?? '', 'disk_capacity');
            $data['display_name']   = $response['displayName'] ?? '';
            $data['last_checkin']   = $response['lastCheckin'] ? format_ms_timestamp_to_db_datetime($response['lastCheckin']) : '';
            $data['model']          = $response['deviceModel'] ?? '';
            $data['imei']           = $response['imei'] ?? '';
            $data['iccid']          = $response['iccid'] ?? '';
            $data['mac_address']    = $response['wifiMacAddress'] ?? '';
            $data['enrolled_date']  = $response['lastRegistrationTime'] ? format_ms_timestamp_to_db_datetime($response['lastRegistrationTime']) : '';
            $data['ownership']      = $response['ownershipType'] ?? '';
            $data['phone_number']   = $response['phoneNumber'] ?? '';
            $data['carrier']        = $response['currentCarrierNetwork'] ?? '';
            $data['platform']       = $response['platformType'] ?? '';
            $data['created_at']     = \Carbon\Carbon::now()->format('Y-m-d H:i:s');
            $data['updated_at']     = \Carbon\Carbon::now()->format('Y-m-d H:i:s');

            return $data;
        }

        return false;
    }


    /**
     * GET Techspec string for finding tech specs
     * @param mixed $data
     *
     * @return [type]
     */
    public function getTechSpecString($data)
    {

        return convertSpecsToReadableValue($data['totalCapacity'] ?? '', 'disk_capacity');
    }

    /**
     *  Update Dicovery tool table asset ID
     * @param mixed $data
     * @param mixed $assetId
     *
     * @return [type]
     */
    public function updateDiscoveryAssetId($data, $assetId)
    {
        return MobileIron::find($data['mobile_iron_id'])->update(['asset_id' => $assetId]);
    }

    /**
     * Auto re assigns the discreapancy assets.
     * @return [type]
     */
    public function reassignAssets()
    {
        $discrepancyDevices = $this->repository->getUserMismatchDevices()->has('user')->get();
        $this->reAssignDiscrepancyAssets($discrepancyDevices, 'MobileIron');
        return true;
    }
}
