<?php

namespace App\Services\Integrations;

use Illuminate\Http\Client\RequestException;
use App\Models\KandjiDevices;
use Illuminate\Support\Facades\Http;
use App\Models\ApiCredential;
use GuzzleHttp\Client;
use App\Http\Traits\AutoCreateAssignTrait;
use App\Models\AssetStatus;
use App\Models\AssetType;
use App\Repositories\DiscoveryTools\KandjiRepository;
use App\Services\DiscoveryTools\Kandji\KandjiSyncService;
use GuzzleHttp\Psr7;
use Exception;
use App\User;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

/**
 * Service Class for MobileIron API
 */
class KandjiIntegration
{
    use AutoCreateAssignTrait;
    protected $api;
    protected $baseUrl;
    protected $userName;
    protected $password;
    protected $apiKey;
    public function __construct(protected KandjiSyncService $service, protected KandjiRepository $repository)
    {
        $this->api      = ApiCredential::where('slug', 'kandji')->first();
        $this->baseUrl  = optional($this->api)->url;
        $this->userName = optional($this->api)->user_name;
        $this->password = optional($this->api)->password;
        $this->apiKey   = optional($this->api)->key;
    }

    /**
     * Check API Connection status
     */
    public function checkApiConnection()
    {
        try {
            $response = $response = Http::withToken($this->apiKey)
                ->retry(2, 1000)
                ->get($this->baseUrl . 'api/v1/devices/', ['limit' => '1'])
                ->throw();

            return $response->successful();
        } catch (Exception $e) {
            Log::channel('single')->info("Kandji API Connection Error :- " . $e->getMessage());
            return false;
        }
    }

    /**
     * Upload the KandjiAPI data to Database
     */
    public function upload()
    {
        $page = 0;
        while (1 == 1) {
            $response   = $this->getAllDevicesData($page);
            if ($response) {
                $response   = $this->saveKandjiData($response);
            } else {
                break;
            }
            //echo $page.'-';
            $page++;
        }

        return true;
    }

    /**
     * Call the API and return the reponse
     *
     * @param int $page  page number
     * @return Array
     */
    public function getAllDevicesData($page = 0)
    {
        $limit  = 300;
        $offset = $page * $limit;
        try {
            $response = Http::withToken($this->apiKey)
                ->retry(3, 3000)
                ->get($this->baseUrl . 'api/v1/devices/', ['offset' => $offset])
                ->throw()
                ->json();
            if ($response) {
                return $response;
            }
        } catch (RequestException $e) {
            Log::channel('single')->info("Kandji Error :- " . $e->getMessage());
        }

        return false;
    }

    /**
     * Call the API and return the reponse
     *
     * @param int $page  page number
     * @return Array
     */
    public function getSingleDevicesData($deviceId)
    {
        try {
            $response = Http::withToken($this->apiKey)
                ->retry(3, 3000)
                ->get($this->baseUrl . "api/v1/devices/$deviceId/details")
                ->throw()
                ->json();

            if ($response) {
                return $response;
            }
        } catch (RequestException $e) {
            Log::channel('single')->info("Kandji Device Details API ($deviceId) Error :- " . $e->getMessage());
        }

        return false;
    }

    /**
     * Save the MobileIron data to DB.
     *
     * @param Array $response  APi response
     *
     * Return boolean
     */

    public function saveKandjiData($response)
    {
        foreach ($response as $key => $data) {
            $deviceDetails = $this->getSingleDevicesData($data['device_id']);

            if ($deviceDetails) {
                $dataToUpdate = [];
                $dataToUpdate = $this->service->parseKandjiApiResponse($deviceDetails, $data);

                $kandjiDeviceData = KandjiDevices::updateOrCreate(
                    ['serial_no' => $dataToUpdate['serial_no']],
                    $dataToUpdate
                );

                $assetTypeId   = AssetType::firstOrCreate(['name' => 'Computer', 'slug' => 'computer'])->id;
                $assetStatusId = AssetStatus::firstOrCreate(['name' => 'Assigned', 'slug' => 'assigned'])->id;
                $dataToUpdate['user_id'] = $this->service->getUserId($data['user']['email'] ?? null);
                $dataToUpdate['asset_type_id'] = $assetTypeId;

                //Auto create and assign asset if asset is not exist
                if ($dataToUpdate['asset_id'] == 0 && $dataToUpdate['user_id'] != 0) {
                    $dataToUpdate['manufacturer'] = 'Apple';
                    $techSpec = $this->getTechSpec($dataToUpdate, 'kandji_model');
                    $dataToUpdate['asset_status_id'] = $assetStatusId;
                    $dataToUpdate['kandji_id']       = $kandjiDeviceData->id;

                    $this->autoCreateAndAssign($dataToUpdate, $techSpec, "Kandji");
                }
            }
        }

        sleep(1);

        return true;
    }

    /** 
     * GET Techspec string for finding tech specs
     * @param mixed $data
     * 
     * @return [type]
     */
    public function getTechSpecString($data)
    {
        if (!$data['processor_speed'] && !$data['ram'] && !$data['disk_capacity']) {
            return 'Ghz/GB/GB';
        }

        return ($data['processor_speed'] ?: 'Ghz') . '/' . ($data['ram'] ?: 'GB') . '/' . ($data['disk_capacity'] ?: 'GB');
    }

    /**
     *  Update Dicovery tool table asset ID
     * @param mixed $data
     * @param mixed $assetId
     * 
     * @return [type]
     */
    public function updateDiscoveryAssetId($data, $assetId)
    {
        return KandjiDevices::find($data['kandji_id'])->update(['asset_id' => $assetId]);
    }


    /**
     * Auto re assigns the discreapancy assets.
     * @return [type] 
     */
    public function autoReassignAssets()
    {
        $discrepancyDevices = $this->repository->getUserMismatchDevices()->has('user')->get();
        $this->reAssignAssets($discrepancyDevices, 'Kandji');
        return true;
    }
}
