<?php

namespace App\Services\Integrations;

use Exception;
use Microsoft\Graph\Graph;
use App\Models\ApiCredential;
use Illuminate\Support\Facades\Log;
use GuzzleHttp\Exception\BadResponseException;
use League\OAuth2\Client\Provider\GenericProvider;

class AzureDirectoryIntegration
{

    /**
     * Retrieve the existing Azure Directory API credential from the database.
     *
     * If the credential does not exist, returns null.
     *
     * @return ApiCredential|null The existing Azure Directory API credential.
     */
    public function getCredentials()
    {
        try {
            return ApiCredential::where('slug', 'azure_directory')->first();
        } catch (Exception $e) {
            Log::error('Failed to load Azure credentials', ['error' => $e->getMessage()]);
            return null;
        }
    }

    /**
     * Gets the OAuth2 client instance to use with the Azure Active Directory
     * integration.
     *
     * This method returns an instance of the `League\OAuth2\Client\Provider\GenericProvider`
     * class.
     *
     * @return \League\OAuth2\Client\Provider\GenericProvider The OAuth2 client instance.
     */
    public function getClient()
    {
        $credentials = $this->getCredentials();
        return new GenericProvider([
            'clientId'                  => optional($credentials)->user_name,
            'clientSecret'              => optional($credentials)->password,
            'redirectUri'               => url('azure-directory-callback'),
            'urlAuthorize'              => config('services.active_directory.base_url') . '/' . optional($credentials)->url . '/oauth2/v2.0/authorize',
            'urlAccessToken'            => config('services.active_directory.base_url') . '/' . optional($credentials)->url . '/oauth2/v2.0/token',
            'urlResourceOwnerDetails'   => '',
            'scopes'                    => config('services.active_directory.scopes')
        ]);
    }

    /**
     * Returns the authorization URL used to authenticate with Azure Active Directory.
     *
     * The authorization URL is the URL that the user needs to visit to authorize the
     * Azure Active Directory integration.
     *
     * @return string The authorization URL.
     */
    public function getAuthUrl()
    {
        $client     = $this->getClient();
        $authUrl    = $client->getAuthorizationUrl();

        session(['oauthState' => $client->getState()]);

        return $authUrl;
    }

    /**
     * Retrieves the access token using the provided authorization code.
     *
     * @param mixed $authCode The authorization code used to retrieve the access token.
     * 
     * @return mixed The access token.
     */
    public function getAccessToken($authCode)
    {
        $client = $this->getClient();
        return $client->getAccessToken('authorization_code', [
            'code' => $authCode
        ]);
    }

    /**
     * Retrieves the current access token.
     *
     * @return string The current access token, or an empty string if it cannot be retrieved.
     */
    public function getCurrentAccessToken()
    {
        $credentials = $this->getCredentials();
        if (empty($credentials->data)) {
            return '';
        }

        $tokens = json_decode($credentials->data);
        if (empty($tokens->accessToken) || empty($tokens->refreshToken) || empty($tokens->tokenExpires)) { // Check if tokens exist
            return '';
        }

        $now = time() + 300;
        if ($tokens->tokenExpires <= $now) {
            try {
                $client     = $this->getClient();
                $newToken   = $client->getAccessToken('refresh_token', [
                    'refresh_token' => $tokens->refreshToken
                ]);

                return $newToken->getToken();
            } catch (Exception $e) {
                Log::error('Failed to retrieve access token', ['error' => $e->getMessage()]);
                return '';
            }
        }

        return $tokens->accessToken;
    }

    /**
     * Checks the connection to Azure Directory Sync.
     * 
     * @return mixed Returns true if the connection is successful, or an array with an error message if the connection fails.
     */
    public function checkConnection()
    {
        $accessToken = $this->getCurrentAccessToken();

        if (empty($accessToken)) {
            return ['token_error' => "Invalid Token!. Please reconnect"];
        }

        $graph = new Graph();
        $graph->setAccessToken($accessToken);

        try {
            $graphResult    = $graph->createRequest("GET", '/users')->execute();

            $body   = $graphResult->getBody();
            $result = $body['value'];

            return $result ? true : false;
        } catch (BadResponseException $e) {
            return ['error' => json_decode($e->getResponse()->getBody()->getContents(), true)];
        }
        return false;
    }

    /**
     * Get the users from the Azure Active Directory with pagination.
     * 
     * @param string $nextLink The link to the next page of users.
     * 
     * @return array|false The list of users and the link to the next page of users, or false if an error occurred.
     */
    public function getUsers($nextLink = null)
    {
        $accessToken = $this->getCurrentAccessToken();
        if (empty($accessToken)) {
            return false;
        }

        $graph = new Graph();
        $graph->setAccessToken($accessToken);

        if ($nextLink === null) {
            $nextLink = '/users?$top=100&$select=id,givenName,displayName,surname,identities,city,country,department,employeeId,jobTitle,officeLocation,streetAddress,mail,accountEnabled,employeeType,employeeLeaveDateTime,createdDateTime,employeeHireDate';
        }

        try {
            $graphResult = $graph->createRequest("GET", $nextLink)->execute();
            return [
                'data' => $graphResult->getBody()['value'],
                'nextLink' => $graphResult->getNextLink()
            ];
        } catch (BadResponseException $e) {
            Log::channel('daily')->info("AD Import Error : " . $e->getResponse()->getBody()->getContents());
            return false;
        }
    }

    /**
     * Retrieves user details from Azure Active Directory using the specified user ID.
     *
     * @param string $id The ID of the user to retrieve details for.
     * 
     * @return array The user details if successful, or an error array if a failure occurs.
     *               The error array includes 'token_error' if the access token is invalid,
     *               or 'error' with the response details in case of a bad response.
     */
    public function getUserDetails($id)
    {
        $accessToken = $this->getCurrentAccessToken();
        if (empty($accessToken)) {
            return ['token_error' => "Invalid Token!. Please reconnect"];
        }

        $graph = new Graph();
        $graph->setAccessToken($accessToken);

        try {
            $graphResult = $graph->createRequest("GET", '/users/' . $id . '?$select=id,givenName,displayName,surname,identities,city,country,department,employeeId,jobTitle,officeLocation,streetAddress,mail,accountEnabled')
                ->execute();
            return $graphResult->getBody();
        } catch (BadResponseException $e) {
            return ['error' => json_decode($e->getResponse()->getBody()->getContents(), true)];
        }
    }

    /**
     * Checks if a user is a member of the specified group in Azure Active Directory.
     * 
     * @param string $userId The ID of the user to check.
     * 
     * @return bool True if the user is a member of the specified group, false otherwise.
     */
    public function checkUserGroups($userId)
    {
        $accessToken = $this->getCurrentAccessToken();
        if (empty($accessToken)) {
            Log::channel('daily')->error('Group check failed - missing access token');
            return false;
        }

        $groupName  = config('azure-directory.allowedGroups.name');
        $groupId    = config('azure-directory.allowedGroups.id');

        $graph = new Graph();
        $graph->setAccessToken($accessToken);
        $nextLink = '/users/' . $userId . '/memberOf/?$count=true&$search="displayName:' . $groupName . '"&$orderby=displayName&$select=displayName,id';

        try {
            $graphResult = $graph->createRequest("GET", $nextLink)
                ->addHeaders(array("ConsistencyLevel" => "eventual"))
                ->execute();

            $body = $graphResult->getBody();

            if (!empty($body['value'])) {
                $groupNames = array_column($body['value'], 'displayName');
                $groupIds   = array_column($body['value'], 'id');

                $nameMatch = in_array($groupName, $groupNames);
                $idMatch = in_array($groupId, $groupIds);

                return $nameMatch && $idMatch;
            }

            return false;
        } catch (BadResponseException $e) {
            $errorResponse = $e->getResponse()->getBody()->getContents();
            Log::channel('daily')->error('Group API request failed', [
                'user_id' => $userId,
                'error' => $e->getMessage(),
                'response' => $errorResponse,
                'status_code' => $e->getResponse()->getStatusCode()
            ]);

            return false;
        }
    }
}
