<?php

namespace App\Services\Integrations\Asn;

use App\Events\BulkUpdates;
use App\Models\AsnHardwareMapping;
use App\Models\Asset;
use App\Models\AssetStatus;
use Carbon\Carbon;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Validator;
use PhpOffice\PhpSpreadsheet\Reader\Xlsx;

class CdwImportService extends AbstractAsnImportService
{

    /**
     * Generate CdwData
     */
    public function createAsnData($item, $count = null)
    {
        $createdData = ['assetData' => [], 'error' => []];
        $assetData = [];
        if ($this->isAccessories($item, 'cdw', 'Manufacturer Part Number')) {
            $createdData = ['assetData' => $assetData, 'error' => ['cdwdata' => ['Line No ' . $count . ': The item is mapped as accessory.']]];
            return $createdData;
        }
        if ($this->isServicese($item['Manufacturer'])) {
            $createdData = ['assetData' => $assetData, 'error' => ['cdwdata' => ['Line No ' . $count . ': The item type is services.']]];
            return $createdData;
        }
        if ($this->isUnwantedPart($item['Manufacturer Part Number'], 'cdw')) {
            $createdData = ['assetData' => $assetData, 'error' => ['cdwdata' => ['Line No ' . $count . ': The Manufacturer Part is marked as Unwanted.']]];
            return $createdData;
        }

        $assetData = $this->setAssetArray($item);
        $existing = Asset::where('serial_no', $assetData['serial_no'])->orWhere('asset_tag', $assetData['asset_tag'])->first();
        if ($existing) {
            $createdData = ['assetData' => $assetData, 'error' => ['cdwdata' => ['Line No ' . $count . ': Asset already exists']]];
            $createdData;
        }
        $createdData = ['assetData' => $assetData, 'error' => []];
        return $createdData;
    }

    /**
     *  Asset array to create
     * @param mixed $item
     * 
     * @return [type]
     */
    public function setAssetArray($item)
    {
        $hardware = [];
        if (isset($item['Manufacturer Part Number'])) {
            $hardware = AsnHardwareMapping::where(['provider' => 'cdw', 'part_no' => $item['Manufacturer Part Number']])->first();
        }
        $assetData = $this->getTechDetails($item, $hardware, 'Manufacturer Part Number');
        $inTransit = AssetStatus::where('slug', 'cdw_in_transit')->first();
        $assetData['po_id']             = $item['Customer Purchase Order'] ?? '';
        $assetData['serial_no']         = stripS($item['Serial Number']);
        $assetData['asset_tag']         = $item['Asset Tag'] ?? ($item['Serial Number'] ?? '');
        $assetData['asset_tag']         = stripS($assetData['asset_tag']);
        $assetData['asset_status_id']   = $inTransit ? $inTransit->id : '';
        $assetData['ticket_no']         = $item['Ticket Number'] ?? '';
        $assetData['part_no']       = $item['Manufacturer Part Number'] ?? '';
        $assetData['asset_original_value']  = $item['Cost'] ?? '';
        $assetData['asset_tracking_id']       = $item['Tracking Number'] ? trim($item['Tracking Number']) : null;
        $assetData['location_id'] = null;

        return $assetData;
    }

    /**
     * createAssetHistory
     *
     * @param  mixed $asset
     * @return void
     */
    public function createAssetHistory($asset)
    {
        $inTransit = Cache::remember('CdwTransitAssetStatus', 3600, function () {
            return AssetStatus::where('slug', 'cdw_in_transit')->first();
        });
        $assetHistory = [
            'asset_id' => $asset->id,
            'action' => 'imported_by_vendor',
            'new_asset_status_id' => $inTransit->id,
            'description' =>  __('history.ImportedByVendor', [
                'asset_name' => $asset->serial_no,
                'asset_id' => $asset->id
            ]),
            'created_by' => 'CDW Import',
            'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
        ];
        event(new BulkUpdates($assetHistory));
    }
}
