<?php

namespace App\Services\Integrations\Asn;

use GuzzleHttp\Exception\RequestException;
use App\Models\ApiCredential;
use App\Models\AssetStatus;
use GuzzleHttp\Client;
use App\Models\Asset;
use App\Models\AsnHardwareMapping;
use GuzzleHttp\Psr7;
use App\Models\AsnAccessoriesMapping;
use App\Events\BulkUpdates;
use App\Models\AssetTracking;
use Exception;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class AsnShiIntegration extends AbstractAsnImportService
{

	public function __construct()
	{
		$this->credential = ApiCredential::where('slug', 'asn_shi')->first();
		$this->inTransitStatus = AssetStatus::where('slug', 'shi_in_transit')->first();

		$this->baseUrl    = config('services.asn_shi.base_url');
		$this->userName   = optional($this->credential)->user_name;
		$this->password   = optional($this->credential)->password;
		$this->accountNo   = optional($this->credential)->account_no;
	}

	/**
	 * Check the API is connected or not
	 */
	public function checkConnection()
	{
		$authCode = $this->getAuthCode();
		if ($authCode === false) {
			return false;
		}

		$client = new Client([
			'headers' => [
				'Accept' 		=> 'application/json',
				'Authorization' => "Bearer " . $authCode,
				'Content-Type'  => 'application/json',
			],
		]);

		try {
			$response = $client->get($this->baseUrl . "v2/customer/whoami");
			$response = json_decode($response->getBody()->getContents());

			return $response->ClientID ? true : false;
		} catch (Exception $e) {
			//Log::channel('single')->info("ASN SHI Connection Error :- " . $e->getMessage());

			return false;
		}
	}

	/**
	 * Get the Authorization code
	 */
	public function getAuthCode()
	{
		$client = new Client([
			'headers' => [
				'Accept' => 'application/json',
				'Authorization' => "Basic " . base64_encode($this->userName . ':' . $this->password),
				'Content-Type' => 'application/json',
			],
		]);

		try {
			$param = [
				'form_params' => [
					'grant_type' => 'client_credentials',
					'scope' 	=> config('services.asn_shi.scope')
				]
			];

			$response = $client->post(config('services.asn_shi.token_url'), $param);

			$response = json_decode($response->getBody()->getContents());

			return $response->access_token ?? false;
		} catch (Exception $e) {
			Log::channel('single')->info("ASN SHI Error :- " . $e->getMessage());

			return false;
		}
	}

	/**
	 * Get the Order Details
	 *
	 * @param Date $startDate  Taking the order details from this date
	 */
	public function getOrders($startDate)
	{
		$authCode = $this->getAuthCode();
		if ($authCode === false) {
			return false;
		}
		$client = new Client([
			'headers' => [
				'Accept' 		=> 'application/json',
				'Authorization' => "Bearer " . $authCode,
				'Content-Type'  => 'application/json',
			],
		]);

		try {
			$response = $client->get($this->baseUrl . "v2/customer/Order?TouchedOnBeginUTC=" . $startDate);
			$response = json_decode($response->getBody()->getContents());

			return $response ? $response->Orders : false;
		} catch (RequestException $e) {
			Log::channel('single')->info("ASN SHI Error :- " . Psr7\str($e->getResponse()));
		}

		return false;
	}

	/**
	 * Import the SHI data and save the assets
	 */
	public function importShiDetails($startDate)
	{
		$orders 	= $this->getOrders($startDate);
		if (!$orders) {
			return 0;
		}

		$totalSavedAssets = 0;

		foreach ($orders as $order) {

			$totalSavedAssets += $this->saveAssetDetails($order);
		}

		return $totalSavedAssets;
	}

	/**
	 * Process the SHI API response and save the assets
	 */
	public function saveAssetDetails($order)
	{
		$poNumber    = $order->PONumber;
		$counter     = 0;

		foreach ($order->ItemList as $itemList) {
			$partNo = $itemList->MPN;

			foreach ($itemList->Shipments as $key => $shipment) {
				foreach ($shipment->SerializedItemList as $key => $serializedItemList) {

					$serializedItemList->partNo = $partNo;
					$serializedItemList->poNumber = $poNumber;

					$data = $this->CreateAsnData($serializedItemList, $shipment->TrackingNumber);
					if ($this->validateItem($data)) {
						$existing = Asset::where('serial_no', $serializedItemList->SerialNumber)->first();
						if ($existing) {
							//echo "\nexisting  ".$existing->id;
							continue;
						}
						try {
							$tracking = $this->createAssetTracking($shipment->TrackingNumber, $shipment->Carrier);

							$data['asset_tracking_id'] = $tracking ? $tracking->id : null;

							$asset = Asset::Create($data);
							$this->createAssetHistory($asset);

							$counter++;
						} catch (Exception $e) {
							// dd($e->getMessage());
							//Nothing to do
						}
					}
				}
			}
		}

		return $counter;
	}

	/**
	 * Generate Array for saving data
	 *
	 * @param object $item   SHI API single data response for an Asset
	 * @param String $trackingNo
	 * 
	 * @return Array
	 */
	public function CreateAsnData($item, $trackingNo = null)
	{
		$assetData = [];
		$hardware = [];

		if ($this->isAccessories($item)) {
			return $assetData;
		}

		if ($this->isUnwantedPart($item->partNo, 'shi')) {
			return $assetData;
		}
		if (isset($item->partNo)) {
			$hardware = AsnHardwareMapping::where('provider', 'shi')->where('part_no', 'like', trim($item->partNo))->first();
		}

		$assetData = $this->getTechDetails($item, $hardware);
		$assetData = $this->setAssetArray($item);

		return $assetData;
	}

	/**
	 *  Asset array to create
	 * @param mixed $item
	 * 
	 * @return [type]
	 */
	public function setAssetArray($item)
	{
		$assetData['po_id']         = $item->poNumber ? trim($item->poNumber) : null;
		$assetData['serial_no']         = $item->SerialNumber ? stripS(trim($item->SerialNumber)) : null;
		$assetData['asset_tag']         = !empty($item->AssetTag) ? trim($item->AssetTag) : null;
		$assetData['asset_status_id']   = $this->inTransitStatus ? $this->inTransitStatus->id : null;
		$assetData['part_no']           = $item->partNo ? trim($item->partNo) : null;
		// $assetData['tracking_no']       = $trackingNo;

		return $assetData;
	}

	/**
	 * Check the item is Accessory or based on the Mapping
	 *
	 * @param  $item
	 *
	 * @return Boolean
	 */
	public function isAccessories($item, $provider = 'shi', $manufacturerPart = null)
	{
		if (isset($item->partNo)) {

			$accessories = AsnAccessoriesMapping::where(['provider' => 'shi', 'part_no' => trim($item->partNo)])->first();

			return $accessories ? true : false;
		}

		return false;
	}

	/**
	 * Create the SHI Asset creation History
	 */
	public function createAssetHistory($asset)
	{
		$inTransit = Cache::remember('ShiTransitAssetStatus', 3600, function () {
			return AssetStatus::where('slug', 'shi_in_transit')->first();
		});
		$assetHistory = [
			'asset_id' => $asset->id,
			'action' => 'imported_by_vendor',
			'new_asset_status_id' => $inTransit->id,
			'description' =>  __('history.ImportedByVendor', [
				'asset_name' => $asset->serial_no,
				'asset_id' => $asset->id
			]),
			'created_by' => 'SHI Import',
		];

		//AssetHistory::create($assetHistory);
		event(new BulkUpdates($assetHistory));
	}

	/**
	 * Create the Asset Tracking details and return the id
	 *
	 * @param string $trackingNo   Tracking number
	 * @param string $carrier      Tracking Carrier
	 */
	public function createAssetTracking($trackingNo, $carrier = null)
	{
		$tracking = AssetTracking::firstOrCreate(
			['tracking_number' => $trackingNo],
			['carrier' => $carrier]
		);

		return $tracking;
	}
}
