<?php

namespace App\Services\HardwareMapping;

use App\Models\AirWatch;
use App\Models\Asset;
use Facades\App\Repositories\HardwareStandards;
use App\Models\DiscoveryToolsHardwareMapping;
use App\Models\JamfComputer;
use App\Models\IntuneComputer;
use App\Models\AssetType;
use App\Models\Chromebook;
use App\Models\KandjiDevices;
use App\Models\MobileIron;
use App\Models\TechnicalSpecs;
use Illuminate\Support\Facades\Validator;


class DiscoveryToolsHarwareMappingService
{
    /**
     * Hardware mapping data
     */
    public function getMappingData()
    {
        $assetTypes        = AssetType::whereIn('slug', ['computer', 'laptop', 'tablet'])->pluck('id');
        $teqtivityHardwares = DiscoveryToolsHardwareMapping::with('makeAndModel')->get();
        $hardwares = HardwareStandards::getHardwareStandardsByAssetTypes($assetTypes);


        $intuneHardwares   = IntuneComputer::distinct()->select('model')->pluck('model', 'model');
        $intuneHardwares     = $intuneHardwares->prepend("Select", '');

        $chromeHardwares   = Chromebook::distinct()->select('model')->pluck('model', 'model');
        $chromeHardwares     = $chromeHardwares->prepend("Select", '');

        $jamfHardwares   = JamfComputer::distinct()->select('model')->pluck('model', 'model');
        $jamfHardwares     = $jamfHardwares->prepend("Select", '');

        $workspaceOneHardwaresAll   = AirWatch::distinct()->select('model')->pluck('model', 'model');
        $workspaceOneHardwaresAll     = $workspaceOneHardwaresAll->prepend("Select", '');

        $workspaceOneHardwares = [];
        foreach ($workspaceOneHardwaresAll as $key => $value) {
            $workspaceOneHardwares[str_replace('"', "+", $key)] = $value;
        }
        $workspaceOneHardwares = collect($workspaceOneHardwares);

        $kandjiHardwares   = KandjiDevices::distinct()->select('model')->pluck('model', 'model');
        $kandjiHardwares     = $kandjiHardwares->prepend("Select", '');

        $mobileIronHardwares   = MobileIron::distinct()->select('model')->pluck('model', 'model');
        $mobileIronHardwares     = $mobileIronHardwares->prepend("Select", '');

        $discoveryToolMappingData  = DiscoveryToolsHardwareMapping::all();

        $intuneMappedHardwares = $discoveryToolMappingData->where('mdm_name', 'intune_model')->pluck('mdm_model', 'make_and_model_id');
        $chromeMappedHardwares = $discoveryToolMappingData->where('mdm_name', 'chrome_model')->pluck('mdm_model', 'make_and_model_id');
        $jamfMappedHardwares = $discoveryToolMappingData->where('mdm_name', 'jamf_model')->pluck('mdm_model', 'make_and_model_id');
        $workspaceOneMappedHardwares = $discoveryToolMappingData->where('mdm_name', 'workspace_one_model')->pluck('mdm_model', 'make_and_model_id');
        $kandjiMappedHardwares = $discoveryToolMappingData->where('mdm_name', 'kandji_model')->pluck('mdm_model', 'make_and_model_id');
        $mobileIronMappedHardwares = $discoveryToolMappingData->where('mdm_name', 'mobile_iron_model')->pluck('mdm_model', 'make_and_model_id');

        return compact('teqtivityHardwares', 'hardwares',  'intuneHardwares', 'chromeHardwares', 'jamfHardwares', 'workspaceOneHardwares', 'kandjiHardwares', 'mobileIronHardwares', 'discoveryToolMappingData', 'intuneMappedHardwares', 'chromeMappedHardwares', 'jamfMappedHardwares', 'workspaceOneMappedHardwares', 'kandjiMappedHardwares', 'mobileIronMappedHardwares');
    }

    /**Get mapping devices
     * @return [type]
     */
    public function getMappingDevices()
    {
        $chromeComputer = Chromebook::whereNotNull('model')->groupBy('model')->get();
        $intuneComputer = IntuneComputer::whereNotNull('model')->groupBy('model')->get();
        $jamfComputer = JamfComputer::whereNotNull('model')->groupBy('model')->get();
        $workspaceOneComputer = AirWatch::whereNotNull('model')->groupBy('model')->get();
        $kandjiComputer = KandjiDevices::whereNotNull('model')->groupBy('model')->get();
        $mobileIrons = MobileIron::whereNotNull('model')->groupBy('model')->get();
        return compact('chromeComputer', 'intuneComputer', 'jamfComputer', 'workspaceOneComputer', 'kandjiComputer', 'mobileIrons');
    }

    /** Get mapped devices
     * @return [type]
     */
    public function getMappedDevices()
    {
        $mappedChrome = DiscoveryToolsHardwareMapping::where('mdm_name', 'chrome_model')->groupBy('mdm_model')
            ->pluck('mdm_model')->all();
        $mappedIntune = DiscoveryToolsHardwareMapping::where('mdm_name', 'intune_model')->groupBy('mdm_model')
            ->pluck('mdm_model')->all();
        $mappedJamf = DiscoveryToolsHardwareMapping::where('mdm_name', 'jamf_model')->groupBy('mdm_model')
            ->pluck('mdm_model')->all();
        $mappedWorkspaceOne = DiscoveryToolsHardwareMapping::where('mdm_name', 'workspace_one_model')->groupBy('mdm_model')
            ->pluck('mdm_model')->all();
        $mappedKandji = DiscoveryToolsHardwareMapping::where('mdm_name', 'kandji_model')->groupBy('mdm_model')
            ->pluck('mdm_model')->all();
        $mappedMobileIrons = DiscoveryToolsHardwareMapping::where('mdm_name', 'mobile_iron_model')->groupBy('mdm_model')
            ->pluck('mdm_model')->all();
        return compact('mappedChrome', 'mappedIntune', 'mappedJamf', 'mappedWorkspaceOne', 'mappedKandji', 'mappedMobileIrons');
    }

    /** Search filter
     * @return [type]
     */
    public function getMappingSearchData()
    {
        $assetTypes        = AssetType::whereIn('slug', ['computer', 'laptop'])->pluck('id');
        $teqtivityHardwares = DiscoveryToolsHardwareMapping::with('makeAndModel');
        /**
         * if there is filter, then show the filtered teqtivity hardwares only
         */
        if (request('make_and_model')) {
            $teqtivityHardwares = $teqtivityHardwares->whereIn('make_and_model_id', request('make_and_model'));
        }

        if (request('mdm_name')) {
            $teqtivityHardwares = $teqtivityHardwares->whereIn('mdm_name', request('mdm_name'));
        }

        $teqtivityHardwares = $teqtivityHardwares->get();

        return compact('teqtivityHardwares');
    }

    /** 
     * Filter the query
     */
    public function filterMappedData()
    {
        $teqtivityHardwareFilter = request('make_and_model');
        $intuneHardwaresFilter   = request('make_and_model_intune');

        $discoveryToolMappingData  = DiscoveryToolsHardwareMapping::query();

        if ($teqtivityHardwareFilter) {
            $discoveryToolMappingData = $discoveryToolMappingData->whereIn('make_and_model_id', $teqtivityHardwareFilter);
        }

        if ($intuneHardwaresFilter) {
            $discoveryToolMappingData = $discoveryToolMappingData->whereIn('intune_model', $intuneHardwaresFilter);
        }

        return $discoveryToolMappingData->get();
    }

    /** Functions performs the mapping operations
     * @param mixed $discoveryComputers
     * @param mixed $mdmName
     * 
     * @return [type]
     */
    public function mapDiscoveryHardware($discoveryComputers, $mdmName)
    {

        foreach ($discoveryComputers as $key => $discoveryComputer) {
            $makeAndModelMapping = DiscoveryToolsHardwareMapping::select('make_and_model_id')
                ->with('makeAndModel:id,asset_type_id')
                ->where(['mdm_name' => $mdmName, 'mdm_model' => $discoveryComputer->model])->first();

            $techSpec = $this->getTechSpec($discoveryComputer, $mdmName);

            $technicalSpecs = TechnicalSpecs::firstOrCreate([
                'details' => $techSpec,
                'make_and_model_id' => $makeAndModelMapping->make_and_model_id,
            ], [
                'original_value' => 0,
                'original_value' => 0,
                'auto_created' => 1,
            ]);
            Asset::where('id', $discoveryComputer->asset_id)
                ->update([
                    'asset_type_id' => optional($makeAndModelMapping->makeAndModel)->asset_type_id,
                    'make_and_model_id' => $makeAndModelMapping->make_and_model_id,
                    'technical_spec_id' => $technicalSpecs->id,
                ]);
        }

        return true;
    }

    /** Gets tech spec based on discovery tool
     * @param mixed $discoveryComputer
     * @param mixed $deviceType
     * 
     * @return [type]
     */
    public function getTechSpec($discoveryComputer, $deviceType)
    {
        $techSpec = '';
        switch ($deviceType) {
            case 'intune_model':
                $techSpec = 'Ghz/GB/' . convertSpecsToReadableValue($discoveryComputer->disk_capacity, 'disk_capacity');
                break;
            case 'jamf_model':
                $techSpec = $discoveryComputer->processor_speed . ' / ' . $discoveryComputer->ram . ' / ' . $discoveryComputer->disk_capacity;
                break;
            case 'chrome_model':
                $techSpec = 'Ghz/GB';
                break;
            case 'workspace_one_model':
                $techSpec = 'Ghz/GB';
                break;
            default:
                $techSpec = '';
        }
        return $techSpec;
    }

    /** Mapping for Jamf
     * @return [type]
     */
    public function mapJamf()
    {
        $mappedJamf = DiscoveryToolsHardwareMapping::where('mdm_name', 'jamf_model')->groupBy('mdm_model')
            ->pluck('mdm_model')->all();
        $jamfComputers = JamfComputer::whereIn('model', $mappedJamf)->whereNotNull('asset_id')->where('asset_id', '!=', '0')->get();

        return $this->mapDiscoveryHardware($jamfComputers, 'jamf_model');
    }

    /** Map intune
     * @return [type]
     */
    public function mapIntune()
    {
        $mappedIntune = DiscoveryToolsHardwareMapping::where('mdm_name', 'intune_model')->groupBy('mdm_model')
            ->pluck('mdm_model')->all();

        $intuneComputers = IntuneComputer::select('model', 'disk_capacity', 'asset_id')
            ->whereIn('model', $mappedIntune)->whereNotNull('asset_id')->where('asset_id', '!=', '0')->get();

        return $this->mapDiscoveryHardware($intuneComputers, 'intune_model');
    }

    /** map Chrome Book
     * @return [type]
     */
    public function mapChromebook()
    {
        $mappedChromeBook = DiscoveryToolsHardwareMapping::where('mdm_name', 'chrome_model')->groupBy('mdm_model')
            ->pluck('mdm_model')->all();
        $chromeDevices = Chromebook::whereIn('model', $mappedChromeBook)->whereNotNull('asset_id')->where('asset_id', '!=', '0')->get();

        return $this->mapDiscoveryHardware($chromeDevices, 'chrome_model');
    }

    /** map Workspace ONE
     * @return [type]
     */
    public function mapWorkspaceOne()
    {
        $mappedWorkspaceOne = DiscoveryToolsHardwareMapping::where('mdm_name', 'workspace_one_model')->groupBy('mdm_model')
            ->pluck('mdm_model')->all();

        $workspaceOneComputers = AirWatch::whereIn('model', $mappedWorkspaceOne)->whereNotNull('asset_id')->where('asset_id', '!=', '0')->get();

        return $this->mapDiscoveryHardware($workspaceOneComputers, 'workspace_one_model');
    }

    /** map Kandji
     * @return [type]
     */
    public function mapKandji()
    {
        $mappedChromeBook = DiscoveryToolsHardwareMapping::where('mdm_name', 'kandji_model')->groupBy('mdm_model')
            ->pluck('mdm_model')->all();
        $chromeDevices = KandjiDevices::whereIn('model', $mappedChromeBook)->whereNotNull('asset_id')->where('asset_id', '!=', '0')->get();

        return $this->mapDiscoveryHardware($chromeDevices, 'kandji_model');
    }

    /** map MobileIron
     * @return [type]
     */
    public function mapMobileIron()
    {
        $mappedWorkspaceOne = DiscoveryToolsHardwareMapping::where('mdm_name', 'mobile_iron_model')->groupBy('mdm_model')
            ->pluck('mdm_model')->all();

        $workspaceOneComputers = MobileIron::whereIn('model', $mappedWorkspaceOne)->whereNotNull('asset_id')->where('asset_id', '!=', '0')->get();

        return $this->mapDiscoveryHardware($workspaceOneComputers, 'mobile_iron_model');
    }
}
