<?php

namespace App\Services\GoogleWorkspace;

use App\Events\BulkUpdates;
use App\Models\Location;
use App\Models\LocationType;
use App\Models\SiteCode;
use App\Services\Integrations\GoogleWorkspace\GoogleWorkspaceIntegration;
use Carbon\Carbon;
use Exception;
use Firebase\JWT\JWT;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class GsuiteLocationService
{
    public function __construct(protected GoogleWorkspaceIntegration $integration) {}

    /**
     * Retrieves the JWT access token for the Google Directory API
     *
     * @return string|false The JWT access token or false if an error occurred
     */
    public function getJwtAccessToken()
    {
        return $this->integration->getAccessTokenWithJwt('gsuite');
    }

    /**
     * Import the Gsuite locations
     *
     * @return void
     */
    public function importGsuiteLocations()
    {
        $token = $this->getJwtAccessToken();
        $pageToken = '';
        $savedLocations = [];
        $count = 0;

        while (1) {

            $response = $this->integration->getGsuiteLocations($token, $pageToken);

            if ($response === false) {
                return null;
            }

            if (!empty($response['nextPageToken'])) {
                $pageToken = $response['nextPageToken'];
            }

            // if (isset($response['items'])) {
            //     $savedLocations = array_merge($savedLocations, $this->saveLocations($response['items']));

            //     $count += count($savedLocations);
            // }

            $savedLocations = $this->saveLocations($response['items']);
            $count += count($savedLocations);

            if ($pageToken == '') {
                break;
            }
        }

        // if ($savedLocations) {
        //     $this->deactiveLocation($savedLocations);
        // }

        Cache::forget('install-locations');
        Cache::forget('warehouse-locations');

        return $count;
    }

    /**
     * Save the given Gsuite locations
     *
     * @param array $locations The Gsuite locations to save
     *
     * @return array The IDs of the saved locations
     */
    public function saveLocations($locations)
    {
        $savedLocations = [];

        if (!empty($locations)) {

            foreach ($locations as $item) {
                try {
                    $savedLocations[] = $this->saveLocation($item);
                } catch (\Exception) {
                    Log::channel('daily')->info($item['resourceId'] . $item['resourceName'] . " Cannot be saved .");
                }
            }
        }

        return $savedLocations;
    }

    /**
     * Saves the location of an item.
     *
     * @param mixed $item The item to save the location for.
     * @throws Some_Exception_Class If there is an error saving the location.
     * @return int The ID of the saved location.
     */
    public function saveLocation($item)
    {
        $locationData = $this->getLocationData($item);

        $roomName = $locationData['room_name'];

        $this->createLocationHistory($item, $roomName);

        $locationId = $this->createLocation($item, $locationData, $roomName);

        return $locationId;
    }

    /**
     * Retrieves the location data for a given item.
     *
     * @param mixed $item The item to get the location data for.
     * @return array The location data for the item.
     */
    private function getLocationData($item)
    {
        return [
            'room_name' => $item['resourceName'],
            'room_capacity' => $item['capacity'] ?? null,
            'address' => $item['generatedResourceName'],
            'floor' => 0
        ];
    }

    /**
     * Create a history record when a location is created from a GSuite resource.
     *
     * @param array $item The GSuite resource item.
     * @param string $roomName The name of the room.
     *
     * @return boolean
     */
    private function createLocationHistory($item, $roomName)
    {
        if (Location::where('resource_id', $item['resourceId'])->count() == 0) {
            $description = __('history.LocationCreated', [
                'roomName' => $roomName
            ]);
            $assetHistory = [
                'action' => 'location_added',
                'description' => $description,
                'created_at'  => Carbon::now()->format('Y-m-d H:i:s'),
                'updated_at'  => Carbon::now()->format('Y-m-d H:i:s'),
                'created_by' => 'GSuite',
            ];
            event(new BulkUpdates($assetHistory));
        }

        return true;
    }

    /**
     * Creates a location record based on a GSuite resource.
     *
     * @param array $item The GSuite resource item.
     * @param array $locationData The location data array.
     * @param string $roomName The name of the room.
     *
     * @return int The ID of the created or updated location.
     */
    private function createLocation($item, $locationData, $roomName)
    {
        $locationTypeId = Cache::remember('location_type_install_id', 60 * 60, function () {
            return LocationType::where('location_type_name', 'Install')->first()->id;
        });

        $siteCode = Cache::remember('site_code_' . ($locationData['siteCode'] ?? 0), 60 * 60, function () use ($locationData) {
            return SiteCode::where('code', $locationData['siteCode'] ?? 0)->first();
        });

        return Location::updateOrCreate(
            ['resource_id' => $item['resourceId']],
            [
                'room_name'         => $roomName,
                'location_type_id'  => $locationTypeId,
                'site_code_id'      => $siteCode,
                'address'           => $locationData['address'] ?? null,
                'building'          => $locationData['building'] ?? null,
                'floor'             => $locationData['floor'] ?? null,
                'room_capacity'     => $locationData['room_capacity'] ?? null,
                'vc'                => $locationData['vc'] ?? null,
                'status'            => 1,
                'region_id'         => 0,
                'country_id'        => 0,
            ]
        )->id;
    }

    /**
     * Deactivates a location.
     *
     * @param mixed $location The location to be deactivated.
     * @throws \Some_Exception_Class If an exception occurs during deactivation.
     * @return mixed The number of rows affected by the deactivation.
     */
    public function deactiveLocation($location)
    {
        return Location::where('manually_created', 0)->whereNotIn('id', $location)->update(['status' => 0]);
    }
}
