<?php

namespace App\Services;

use App\Repositories\CommonFilterRepository;
use App\Repositories\FilterFieldRepository;

class FilterFieldService
{
    protected $repository;
    protected $commonFilterRepository;

    /**
     * Constructor for initializing the FilterFieldRepository and CommonFilterRepository.
     *
     * @param FilterFieldRepository $repository The FilterFieldRepository instance
     * @param CommonFilterRepository $commonFilterRepository The CommonFilterRepository instance
     */
    public function __construct(FilterFieldRepository $repository, CommonFilterRepository $commonFilterRepository)
    {
        $this->repository = $repository;
        $this->commonFilterRepository = $commonFilterRepository;
    }

    /**
     * Search for teqtivity users
     *
     * @param string $search  search value
     *
     * @return Array
     */
    public function getFilterTeqtivityUser($search)
    {
        $res = $this->repository->getTeqtivityUsers($search);

        $result = $res->mapWithKeys(function ($item, $key) {
            return [
                $key => [
                    'id' => $item->id,
                    'text' => $item->first_name . ' ' . $item->last_name . ' - ' . $item->email,
                ]
            ];
        });
        return $result;
    }

    /**
     * Get hardware standards based on asset type and manufacturer and returned the processed data for dropdown
     *
     * @return array
     */
    public function getHardwareStandards(): array
    {
        $inputData = request()->all();
        $query = $this->repository->getHardwareStandardData();
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'asset_type_id', $inputData['asset_type'] ?? '');
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'manufacturer_id', $inputData['manufacturer'] ?? '');
        $result = $query->get();
        $data = $this->getOutputData($result, 'id', 'name');

        return $data;
    }

    /**
     * Get hardware standards based on asset type, make and model and manufacturer and returned the processed data for dropdown
     *
     * @return array
     */
    public function getTechnicalSpecs(): array
    {
        $inputData = request()->all();
        $query = $this->repository->getTechnicalSpecData();
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'make_and_model_id', $inputData['make_and_model'] ?? '');
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'asset_type_id', $inputData['asset_type'] ?? '');
        $query = $this->commonFilterRepository->filterWithDirectFields($query, 'manufacturer_id', $inputData['manufacturer'] ?? '');
        $result = $query->get();
        $data = $this->getOutputData($result, 'id', 'name');

        return $data;
    }

    /**
     * Get the output data from the given result using the specified key and value attributes.
     *
     * @param mixed $result The input result data
     * @param string $keyAttribute The key attribute to be used
     * @param string $valeAttribute The value attribute to be used
     * @return array The output data
     */
    public function getOutputData($result, $keyAttribute, $valeAttribute): array
    {
        $data = [];

        foreach ($result as $row) {
            $data[] = [
                'id' => $row->{$keyAttribute},
                'name' => $row->{$valeAttribute},
            ];
        }

        return $data;
    }

    /**
     * Retrieves a list of filtered users with specific fields based on the given search term.
     *
     * @param string $search The search term to filter the users.
     *
     * @return array Returns an array of users with their corresponding IDs and concatenated fields.
     */
    public function getFilterUserWithTeqtivityUsersFields($search)
    {
        $query = $this->repository->getAllUsers();
        $query = $this->commonFilterRepository->searchToMultipleFields($query, ['first_name', 'last_name', 'email'], $search);
        $query = $query->limit(15)->get();

        $result = $query->mapWithKeys(function ($item, $key) {
            return [
                $key => [
                    'id' => $item['id'],
                    'text' => $item['first_name'] . ' ' . $item['last_name'] . ' - ' . $item['email'],
                ]
            ];
        });

        return $result;
    }
}
