<?php

namespace App\Services;

use App\Models\FailedTicketHistory as FailedTicketHistoryModal;

class FailedTicketHistoryService
{

    /**
     * Retrieves the data for failed ticket history.
     *
     * @return array Returns an array containing the failed ticket history and the count.
     */
    public function data()
    {

        $failedTicketHistory = FailedTicketHistoryModal::with(['user']);

        $start = request('start');
        $limit = request('length');
        $count = $failedTicketHistory->count();
        if ($limit != -1) {
            $failedTicketHistory = $failedTicketHistory->offset($start)
                ->limit($limit);
        }

        $failedTicketHistory = $failedTicketHistory->orderBy('created_at', 'desc');
        $failedTicketHistory = $failedTicketHistory->get();
        return compact('failedTicketHistory', 'count');
    }

    /**
     * Retrieves the nested data for failed tickets.
     *
     * @param array $failedTicket The array of failed ticket objects.
     * @param int $start The starting index for the nested data.
     * @param array $data The array of existing data to append the nested data to.
     * @return array The updated array of data with the nested data appended.
     */
    public function getNestedData($failedTicket, $start, $data)
    {
        foreach ($failedTicket as $key => $history) {
            $nestedData['id'] = $start + ($key + 1);
            $nestedData['ticket'] = $history->ticket_id;
            $nestedData['created_at'] = $history->created_date;
            $nestedData['user_name'] = $history->user ? $history->user->first_name . " " . $history->user->last_name  : '';
            $nestedData['description'] = $history->description;

            $data[] = $nestedData;
        }
        return $data;
    }

    /**
     * Retrieves export data from the FailedTicketHistoryModal table.
     *
     * @return Collection A collection of FailedTicketHistoryModal records.
     */
    public function getExportData()
    {
        $failedTicketHistory = FailedTicketHistoryModal::orderBy('created_at', 'desc')->get();

        return $failedTicketHistory;
    }

    /**
     * Generates nested export data from failed tickets.
     *
     * @param array $failedTicket The array of failed tickets.
     * @param int $start The starting index for numbering the nested data.
     * @param array $data The array of data to store the nested data.
     * @return array The updated array of data with nested export data.
     */
    public function getExportNestedData($failedTicket, $start, $data)
    {
        foreach ($failedTicket as $key => $history) {
            $nestedData['#'] = $start + ($key);
            $nestedData['Ticket #'] = $history->ticket_id;
            $nestedData['Created At'] = $history->created_date;
            $nestedData['User Name'] = $history->user ? $history->user->first_name . " " . $history->user->last_name  : '';
            $nestedData['Description'] = $history->description;

            $data[] = $nestedData;
        }
        return $data;
    }
}
