<?php

namespace App\Services\ExportManagement;

use App\Jobs\Exports\DuplicateAssetsExportJob;
use App\Jobs\Exports\ExportReport;
use App\Jobs\Exports\LocationExportJob;
use App\Jobs\Exports\UserExportJob;
use App\Jobs\Exports\ZoomApiDataExportJob;
use App\Models\FileExport;
use Carbon\Carbon;
use Illuminate\Support\Facades\File;
use App\Services\Reports\DuplicateAssetsService;
use Illuminate\Support\Facades\Auth;

/**
 * service class for for large data exports
 */
class ExportManagementService
{

    private $duplicatedAssetsService;

    public function __construct(DuplicateAssetsService $duplicatedAssetsService)
    {
        $this->duplicatedAssetsService =  $duplicatedAssetsService;
    }



    /**
     * Passes location export to queue if there are large number of records
     * @param mixed $locationAll
     *
     * @return [type]
     */
    public function locationLargeExport($locationAll)
    {
        // if we not set up the queue server continue normal export
        if (config('queue.default') == 'sync') {
            return false;
        }

        if ($locationAll->count() > 1000) {
            LocationExportJob::dispatch(request()->all())
                ->onQueue('exports');
            // request()->session()->flash('message', 'Your report will be generated in the background. Go to the <a href="/generated-exports/location_report">Reports Page</a> to download the report.');
            return true;
        } else {
            return false;
        }
    }

    /**
     * Passes user export to queue if there are large number of records
     * @param mixed $locationAll
     *
     * @return [type]
     */
    public function userLargeExport($filteredData)
    {
        // if we not set up the queue server continue normal export
        if (config('queue.default') == 'sync') {
            return false;
        }

        if ($filteredData->count() > 1000) {
            UserExportJob::dispatch(request()->all())
                ->onQueue('exports');
            // request()->session()->flash('message', 'Your report will be generated in the background. Go to the <a href="/generated-exports/user_report">Reports Page</a> to download the report.');
            return true;
        } else {
            return false;
        }
    }

    /**
     * Passes report generator export to queue if there are large number of records
     * @param mixed $assetsAll
     *
     * @return [type]
     */
    public function reportsLargeExport($assetsAll)
    {
        // if we not set up the queue server continue normal export
        if (config('queue.default') == 'sync') {
            return false;
        }

        if ($assetsAll->count() > 1000) {
            ExportReport::dispatch(request()->all())
                ->onQueue('exports');
            // request()->session()->flash('message', 'Your report will be generated in the background. Go to the <a target="_blank" href="/generated-exports/generator_report">Reports Page</a> to download the report.');
            return true;
        } else {
            return false;
        }
    }

    /**
     * This function writes the file to path
     * @param mixed $datas
     * @param string $fileName
     *
     * @return [type]
     */
    public function exportForLater($datas, $fileName = 'report_file_')
    {
        if (empty($datas)) {
            return '';
        }

        $path = storage_path('app/public/export_reports');

        if (!File::isDirectory($path)) {
            File::makeDirectory($path, 0755, true, true);
        }
        $filename = $fileName . Carbon::now()->timestamp . '.csv';
        $fileLocation = $path . '/' . $filename;
        $file = fopen($fileLocation, 'w');

        fputcsv($file, array_keys($datas[0][0]));

        foreach ($datas as $data) {
            foreach ($data as $data) {
                fputcsv($file, $data);
            }
        }

        fclose($file);
        return $filename;
    }

    public function duplicatedAssetsLargeExport($assetsAll)
    {
        // if we not set up the queue server continue normal export
        if (config('queue.default') == 'sync') {
            return false;
        }
        if ($assetsAll->count() > 100) {
            DuplicateAssetsExportJob::dispatch(request()->all(), $this->duplicatedAssetsService)
                ->onQueue('exports');
            request()->session()->flash('message', 'Your report will be generated in the background. Go to the <a href="/generated-exports/duplicated_assets_report">Reports Page</a> to download the report.');
            return true;
        } else {
            return false;
        }
    }

    /**
     * Function to dispatch the queue job for Zoom API data download.
     *
     * @param string $reportType  The type of report.
     *
     * @return bool
     */
    public function zoomApiDataExport($reportType)
    {
        $userId = Auth::id();

        $reportId = FileExport::create([
            'report_name' => 'Zoom API Response',
            'user_id'     => $userId,
            'report_type' => 'zoom_api_response',
        ])->id;

        ZoomApiDataExportJob::dispatch($reportId, $reportType);

        return true;
    }
}
