<?php

namespace App\Services\Employee;

use App\Repositories\Employee;
use App\User;
use Illuminate\Support\Facades\Notification;
use App\Notifications\Dispute;
use App\Notifications\DisputeCommon;
use App\Models\Asset;
use App\Repositories\UserRepository;
use Illuminate\Support\Facades\Auth;

class Landing
{
    protected $employeeRepository;

    public function __construct(Employee $employeeRepository, UserRepository $userRepository)
    {
        $this->employeeRepository = $employeeRepository;
        $this->userRepository = $userRepository;
    }

    /**
     * Get all the assets of a user, and his team if manager
     * @param int $userId 
     * @return array
     */
    public function getEmployeeData(int $userId)
    {
        $userDetails = User::find($userId);
        $employeeAssets = $this->employeeRepository->getEmployeeAssets($userId)->get();
        $isManager = $this->employeeRepository->getManagerTeam($userId)->count() > 0 ? 1 : 0;
        $managerActiveTeam = $this->employeeRepository->getManagerActiveTeam($userId)->get();
        $managerInActiveTeam = $this->employeeRepository->getManagerInActiveTeam($userId)->get();
        $userAssetsCountByType = $this->employeeRepository->getEmployeeAssetsCountByType($userId)->get();
        $userAssetsCountByAge = $this->employeeRepository->getEmployeeAssetsCountByAge($userId)->get();
        $donutAssetsCountByTypeData = [];
        $donutAssetsCountByAgeData = [];

        foreach ($userAssetsCountByType as $asset) {
            $donutAssetsCountByTypeData[] = ['label' => $asset->assetType->name, 'value' => $asset->units];
        }

        foreach ($userAssetsCountByAge as $asset) {
            $current_year = date("Y");
            $age = date("Y") - $asset->new_date;
            if ($age <= 0) {
                $label = '0 - 1 Year';
            } else {
                $label = $age . ' - ' . ($age + 1) . ' Years';
            }

            $donutAssetsCountByAgeData[] = ['label' => $label, 'value' => $asset->units];
        }

        return compact('employeeAssets', 'managerActiveTeam', 'managerInActiveTeam', 'userDetails', 'donutAssetsCountByTypeData', 'donutAssetsCountByAgeData', 'isManager');
    }

    /**
     * Get the assets of an employee
     * @param int $userId 
     * @return query
     */
    public function getEmployeeAssets(int $userId)
    {
        return $this->employeeRepository->getEmployeeAssetsByJoin($userId);
    }

    /**
     * Export Active Team
     * @param int $userId 
     * @return csv
     */
    public function exportActiveTeam(int $userId)
    {
        $managerActiveTeam = $this->employeeRepository->getManagerActiveTeam($userId)->get();

        return $this->employeeRepository->exportActiveTeam($managerActiveTeam);
    }

    public function exportInActiveTeam(int $userId)
    {
        $managerInActiveTeam = $this->employeeRepository->getManagerInActiveTeam($userId)->get();

        return $this->employeeRepository->exportInActiveTeam($managerInActiveTeam);
    }

    public function sendDisputeNotification(int $userId, string $disputeDetails, Asset $asset)
    {
        /*Disabled the email notification as per the client request*/
        //$user = User::find($userId);
        //Notification::route('mail', config('mail.disputeMail'))->notify(new Dispute($user, $disputeDetails, $asset));
    }

    public function sendCommonDisputeNotification(int $userId, array $disputeDetails)
    {
        /*Disabled the email notification as per the client request*/
        //$user = User::find($userId);
        //Notification::route('mail', config('mail.disputeMail'))->notify(new DisputeCommon($user, $disputeDetails));
    }

    public function activeFilter()
    {
        $userId = Auth::id();
        $users = $this->employeeRepository->getManagerActiveTeam($userId);

        $count = $users->count('users.id');
        $users = $this->getOutputData($users);
        return compact('users', 'count');
    }

    public function inactiveFilter()
    {
        $userId = Auth::id();
        $users = $this->employeeRepository->getManagerInActiveTeam($userId);
        $count = $users->count('users.id');
        $users = $this->getOutputData($users);
        return compact('users', 'count');
    }

    public function getOutputData($users)
    {
        $start = request('start');
        $limit = request('length');
        if ($limit != -1) {
            $users = $users->offset($start)
                ->limit($limit);
        }

        $users = $users->orderBy('last_name')
            ->orderBy('first_name')
            ->orderBy('email');
        return $users->get();
    }

    public function getUserData($users, $start, $data)
    {
        $parentIndex = $start;

        foreach ($users as $key => $user) {
            $parentId = $user->id;

            $parentIndex++;

            $email = route('users.show', $user->id);

            $nestedData = $this->getNestedData($user, $parentIndex, '', '', $email);

            $data[] = $nestedData;
        }

        return $data;
    }

    public function getNestedData($user, $index, $last, $parent, $email)
    {
        $nestedData['dataLast'] = $last;
        $nestedData['dataId'] = $user->id;
        $nestedData['dataParent'] = $parent;
        $nestedData['dataCount'] = $user->assets_count;

        if ($user->assets_count > 0) {
            $nestedData['details'] = '<a href="#" class="detail"><i class="icon icon-e-add"></i></a>';
        } else {
            $nestedData['details'] = '';
        }

        $nestedData['id'] = $index;
        $nestedData['first_name'] = $user->first_name;
        $nestedData['last_name'] = $user->last_name;
        $nestedData['hire_date'] = $user->hire_date;
        $nestedData['legalhold'] = $user->legalhold == 1 ? 'Yes' : 'No';
        $nestedData['terminated_date'] = $user->terminated_date;
        if ($user->assets_count > 0) {
            $nestedData['email'] = '<a href="#">' . $user->email . '</a>';
        } else {
            $nestedData['email'] = $user->email;
        }
        $nestedData['employee_type'] = $user->employeeType ? $user->employeeType->name : '';
        $nestedData['no_of_assets'] = $user->assets_count;

        return $nestedData;
    }
}
