<?php

namespace App\Services\Employee;

use Carbon\Carbon;
use App\Notifications\TerminatedUsersAtTerminationDate;
use App\Notifications\TerminatedUsersTwoWeeksAfterTerminationDate;
use App\Notifications\TerminatedUsersFourWeeksAfterTerminationDate;
use App\Repositories\Employee;
use Illuminate\Support\Facades\Notification;
use App\Notifications\TerminatedUsersOneDayAfterTerminationDate;
use App\Notifications\TerminatedUsersTwoDaysAfterTerminationDate;

class EmailNotification
{
    protected $employeeRepository;

    public function __construct(Employee $employeeRepository)
    {
        $this->employeeRepository = $employeeRepository;
    }

    public function notifyManagersHavingTerminatedTeamAssetsToday()
    {
        $managers = $this->employeeRepository->getManagersHavingTerminatedTeamAssets(Carbon::now()->toDateString())->get();

        foreach ($managers as $manager) {
            $teamAssets = $this->employeeRepository->getTerminatedTeamAssets($manager->id, Carbon::now()->toDateString())->get();

            if (count($teamAssets)) {
                $manager->notify(new TerminatedUsersAtTerminationDate($manager->userName, $teamAssets));
            }
        }
    }

    public function notifyManagersHavingTerminatedTeamAssetsForTwoWeeks()
    {
        $dateBeforeTwoWeeks = Carbon::now()->subWeeks(2)->toDateString();
        $managers = $this->employeeRepository->getManagersHavingTerminatedTeamAssets($dateBeforeTwoWeeks)->get();

        foreach ($managers as $manager) {
            $teamAssets = $this->employeeRepository->getTerminatedTeamAssets($manager->id, $dateBeforeTwoWeeks)->get();

            if (count($teamAssets)) {
                $manager->notify(new TerminatedUsersTwoWeeksAfterTerminationDate($manager->userName, $teamAssets));
            }
        }
    }

    public function notifyManagersHavingTerminatedTeamAssetsForFourWeeks()
    {
        $dateBeforeFourWeeks = Carbon::now()->subWeeks(4)->toDateString();
        $managers = $this->employeeRepository->getManagersHavingTerminatedTeamAssets($dateBeforeFourWeeks)->get();

        foreach ($managers as $manager) {
            $teamAssets = $this->employeeRepository->getTerminatedTeamAssets($manager->id, $dateBeforeFourWeeks)->get();

            if (count($teamAssets)) {
                $manager->notify(new TerminatedUsersFourWeeksAfterTerminationDate($manager->userName, $teamAssets));

                Notification::route('mail', config('mail.disputeMail'))->notify(new TerminatedUsersFourWeeksAfterTerminationDate('Admin', $teamAssets));
            }
        }
    }

    public function notifyManagersToCollectAssetAfterTermination()
    {
        $dateBeforeOneDay = Carbon::now()->subDay(1)->toDateString();
        $managers = $this->employeeRepository->getManagersHavingTerminatedTeamAssets($dateBeforeOneDay)->get();

        foreach ($managers as $manager) {
            $users = $this->employeeRepository->getTerminatedEmployees($manager->id, $dateBeforeOneDay)->get();

            if (count($users)) {
                foreach ($users as $user) {
                    $manager->notify(new TerminatedUsersOneDayAfterTerminationDate($manager->userName, $user));
                }
            }
        }
    }

    public function notifyManagersToCollectAndLockAssetAfterTermination()
    {
        $dateBeforeTwoDays = Carbon::now()->subDays(2)->toDateString();
        $managers = $this->employeeRepository->getManagersHavingTerminatedTeamAssets($dateBeforeTwoDays)->get();

        foreach ($managers as $manager) {
            $users = $this->employeeRepository->getTerminatedEmployees($manager->id, $dateBeforeTwoDays)->get();

            if (count($users)) {
                foreach ($users as $user) {
                    $manager->notify(new TerminatedUsersTwoDaysAfterTerminationDate($manager->userName, $user));
                }
            }
        }
    }
}
