<?php

namespace App\Services\DiscoveryTools\Zoom;

use Illuminate\Support\Str;

abstract class AbstractZoomService
{
    /**
     * Taking the input data
     *
     * @param Http\Request
     *
     * @return Array
     */
    public function getInputData()
    {
        $requestedData = request()->input('form', request()->all());

        return [
            'serialNo'               => $requestedData['serial_no'] ?? '',
            'assetType'              => $requestedData['asset_type'] ?? '',
            'assetStatus'            => $requestedData['asset_status'] ?? '',
            'manufacturer'           => $requestedData['manufacturer'] ?? '',
            'makeAndModel'           => $requestedData['make_and_model'] ?? '',
            'technicalSpec'          => $requestedData['technical_spec'] ?? '',
            'model'                  => $requestedData['model'] ?? '',
            'macAddress'             => $requestedData['mac_address'] ?? '',
            'phoneNo'                => $requestedData['phone_no'] ?? '',
            'location'               => $requestedData['location'] ?? '',
            'zoomRoomName'           => $requestedData['room_name_zoom'] ?? '',
            'hostName'               => $requestedData['host_name'] ?? '',
        ];
    }

    /**
     * Filer the results based on filters
     *
     * @param $assets DB Query
     *
     * @return Query
     */
    public function filter($assets)
    {
        $inputData  = $this->getInputData();

        $assets     = $assets->RelationData($inputData['serialNo'], 'zoom_devices.serial_no');
        $assets     = $assets->RelationData($inputData['model'], 'zoom_devices.model');
        $assets     = $assets->RelationData($inputData['zoomRoomName'], 'zoom_devices.room_name');
        $assets     = $assets->RelationData($inputData['macAddress'], 'zoom_devices.mac_addresses');
        $assets     = $assets->RelationData($inputData['hostName'], 'zoom_devices.hostname');
        $assets     = $this->repository->filterWithAssetFields($assets, 'asset_type_id', $inputData['assetType']);
        $assets     = $this->repository->filterWithAssetFields($assets, 'asset_status_id', $inputData['assetStatus']);
        $assets     = $this->repository->filterWithAssetFields($assets, 'make_and_model_id', $inputData['makeAndModel']);
        $assets     = $this->repository->filterWithAssetFields($assets, 'technical_spec_id', $inputData['technicalSpec']);
        $assets     = $this->repository->filterWithLocationFields($assets, 'id', $inputData['location']);
        $assets     = $this->repository->filterWithAssetMakeAndModelFields($assets, 'manufacturer_id', $inputData['manufacturer']);

        return $assets;
    }

    /**
     * Generating data table data from query result
     *
     * @param $users Query Result
     * @param $start Starting index
     * @param $data Collection variable that will return
     *
     * @return array
     */
    public function getReportData($users, $start, $data = [])
    {
        $parentIndex = $start;

        foreach ($users as $user) {
            $parentIndex++;

            $nestedData = $this->getNestedData($user, $parentIndex);
            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Creating array for Data table data
     * @param $asset Query Result
     * @param $index Starting index
     * @param $email route for user details
     *
     * @return array
     */
    public function getNestedData($zoom, $index)
    {
        $nestedData = [];
        $commonData = $this->getExportNestedData($zoom);

        foreach ($commonData as $key => $value) {
            $nestedData[Str::slug($key, '_')] = $value;
        }

        $assetLink = $zoom->asset ? generateAssetLink($zoom->asset, $zoom->serial_no) : $zoom->serial_no;
        $teqtivityLocationLink = $zoom->asset ? generateLocationLink(optional($zoom->asset->location)->id, optional($zoom->asset->location)->room_name) : '';
        $zoomLocationLink = $zoom->location ? generateLocationLink($zoom->location->id, $zoom->room_name) : $zoom->room_name;

        $nestedData['id']   = $index;
        $nestedData['serial'] = $assetLink;
        $nestedData['location'] = $teqtivityLocationLink;
        $nestedData['room_name_zoom'] = $zoomLocationLink;

        return $nestedData;
    }

    /**
     * Generating data to export
     * @param $assets Query Result
     * @param $start Starting index
     * @param $data Collection variable that will return
     *
     * @return array
     */
    public function getReportExportData($assets, $data = [])
    {

        foreach ($assets as $asset) {
            $nestedData = $this->getExportNestedData($asset);
            $data[] = $nestedData;
        }
        return $data;
    }

    /**
     * Creating array for Data table data
     * @param $zoom Query Result
     *
     * @return array
     */
    public function getExportNestedData($zoom)
    {
        $nestedData['Serial #']         = disableCSVInjection($zoom->serial_no);
        $nestedData['Asset Status']     = disableCSVInjection($zoom->asset ? optional($zoom->asset->assetStatus)->name : '');
        $nestedData['Location (Teqtivity)'] = disableCSVInjection($zoom->asset ? optional($zoom->asset->location)->room_name : '');
        $nestedData['Room Name (Zoom)'] = disableCSVInjection($zoom->room_name);
        $nestedData['Site Code (Zoom)'] = disableCSVInjection($zoom->site_code);
        $nestedData['Region (Zoom)']    = disableCSVInjection($zoom->region);
        $nestedData['Device Type (Zoom)'] = disableCSVInjection($zoom->device_type);
        $nestedData['Device System (Zoom)']    = disableCSVInjection($zoom->device_system);
        $nestedData['Status (Zoom)']           = disableCSVInjection($zoom->status);
        $nestedData['Host Name (Zoom)']        = disableCSVInjection($zoom->hostname);
        $nestedData['Manufacturer (Zoom)']     = disableCSVInjection($zoom->manufacturer);
        $nestedData['Make and Model (Zoom)']   = disableCSVInjection($zoom->model);
        $nestedData['Firmware (Zoom)']         = disableCSVInjection($zoom->firmware);
        $nestedData['App Version (Zoom)']      = disableCSVInjection($zoom->app_version);
        $nestedData['IP Address (Zoom)']       = disableCSVInjection($zoom->ip_address);
        $nestedData['MAC Addresses (Zoom)']    = disableCSVInjection($zoom->mac_addresses);

        return $nestedData;
    }

    /**
     * Executing the query
     *
     * @param $users Query
     *
     * @return Query Result
     */
    public function getOutputData($assets)
    {
        $start = request('start');
        $limit = request('length');

        if ($limit != -1) {
            $assets = $assets->skip($start)
                ->take($limit);
        }

        $assets = $assets->orderBy('id', 'desc');

        return $assets->get();
    }
}
