<?php

namespace App\Services\DiscoveryTools\MobileIron;

use App\Repositories\DiscoveryTools\MobileIronRepository;
use App\User;

class NotCheckingInNotManaged extends AbstractMobileIronService
{

    public function __construct(protected MobileIronRepository $repository)
    {
    }

    /**
     * Making query for the report
     */
    public function data()
    {

        $inputData  = $this->getInputData();

        $assets     = $this->repository->getNotManaged();
        $assets     = $this->filter($assets, $inputData);
        $count      = $assets->count();
        $assets     = $this->getOutputData($assets);

        return compact('assets', 'count');
    }

    /**
     * Making the query for export
     */
    public function exportData()
    {
        $inputData  = $this->getExportInputData();

        $assets     = $this->repository->getNotManaged();
        $assets     = $this->filter($assets, $inputData);
        $assets     = $assets->orderBy('id', 'desc');

        return $assets;
    }

    /**
     * Filer the results based on filters
     * 
     * @param $assets DB Query
     * @param Array $inputData   Filter data
     * 
     * @return Query
     */
    public function filter($assets, $inputData)
    {

        $assets     = $this->repository->filterByMakeAndModelFields($assets, 'manufacturer_id', $inputData['manufacturer']);

        $assets     = $this->repository->filterByAssetFields($assets, 'serial_no', $inputData['serial_no']);
        $assets     = $this->repository->filterByAssetFields($assets, 'user_id', $inputData['teq_user']);
        $assets     = $this->repository->filterByAssetFields($assets, 'asset_type_id', $inputData['asset_type']);
        $assets     = $this->repository->filterByAssetFields($assets, 'make_and_model_id', $inputData['make_and_model']);
        $assets     = $this->repository->filterByAssetFields($assets, 'location_id', $inputData['location']);
        $assets     = $this->repository->filterByAssetFields($assets, 'asset_status_id', $inputData['status']);


        $assets     = $this->repository->filterByAssetFields($assets, 'host_name', $inputData['host_name']);
        $assets     = $this->repository->filterByAssetFields($assets, 'mac', $inputData['mac_address']);
        $assets     = $this->repository->filterByAssetFields($assets, 'imei', $inputData['imei']);
        $assets     = $this->repository->filterByAssetFields($assets, 'carrier_id', $inputData['carrier']);

        $assets     = $this->repository->filterWithCarrierCountry($assets, $inputData['carrier_country'], 'carrier');

        $assets     = $this->repository->filterWithUserFields($assets, 'status', $inputData['user_status']);
        $assets     = $this->repository->filterWithUserFields($assets, 'department_id', $inputData['user_department']);
        $assets     = $this->repository->filterWithUserFields($assets, 'city', $inputData['work_location']);


        $assets     = $this->repository->filterWithLastModifiedDate($assets, $inputData['last_seen_from'], $inputData['last_seen_to']);

        return $assets;
    }


    /**
     * Creating array for Data table data
     * @param $asset Query Result
     * @param $index Starting index
     * @param $email route for user details
     *
     * @return array
     */
    public function getNestedData($asset, $index)
    {
        $assetLink  = $asset ? route('assets.show', $asset->id) : 'javascript:void(0)';
        $teqtivityUser      = optional($asset->user);
        $teqtivityUserLink  = $teqtivityUser ? route('users.show', $teqtivityUser->id ?: 0) : 'javascript:void(0)';
        $serialNo  = $asset ? ("<a href=$assetLink>" . $asset->serial_no . "</a>") : $asset->serial_no;

        $nestedData['id']               = $index;
        $nestedData['serial_no']        = $serialNo;
        $nestedData['asset_type_teqtivity']     = optional(optional($asset)->assetType)->name;
        $nestedData['manufacturer_teqtivity']     = optional(optional(optional($asset)->makeAndModel)->manufacturer)->name;
        $nestedData['hardware_teqtivity']            = optional(optional($asset)->makeAndModel)->name;
        $nestedData['location_teqtivity']     = optional(optional($asset)->location)->room_name;
        $nestedData['asset_status_teqtivity']     = $asset ? optional($asset->assetStatus)->name : '';

        $nestedData['teqtivity_user'] = $teqtivityUser ? $teqtivityUser->first_name . ' ' . $teqtivityUser->last_name : '';
        $nestedData['teqtivity_user']   = $teqtivityUser ? ("<a href=$teqtivityUserLink>" . $teqtivityUser->first_name . ' - ' . $teqtivityUser->email . "</a>") : '';
        $nestedData['teqtivity_user_status']  = optional(optional($asset)->user)->status == 1 ? 'Active' : 'Terminated';
        $nestedData['teqtivity_user_department']  = optional(optional(optional($asset)->user)->department)->name;
        $nestedData['teqtivity_user_work_location']  = optional(optional($asset)->user)->city;

        $nestedData['host_name_teqtivity']        = $asset->host_name;
        $nestedData['mac_address_teqtivity']      = $asset->mac;
        $nestedData['imei_teqtivity']             = $asset->imei;

        $nestedData['carrier']      = optional($asset->carrier)->name;
        $nestedData['carrier_country']  = optional(optional($asset->mobileCarrier)->country)->name;
        $nestedData['last_modified_date']   = optional($asset)->latestAssetHistory ? optional(optional($asset)->latestAssetHistory)->updated_at : '';

        return $nestedData;
    }

    /**
     * Creating array for export data
     * @param $asset Query Result
     * @param $index Starting index
     *
     * @return array
     */
    public function getExportNestedData($asset, $index, $requiredColumns = null)
    {
        $nestedData['Serial # (Teqtivity)'] = disableCSVInjection($asset->serial_no);
        $nestedData['Asset Type (Teqtivity)'] = optional(optional($asset)->AssetType)->name;
        $nestedData['Manufacturer (Teqtivity)'] = optional(optional(optional($asset)->makeAndModel)->manufacturer)->name;
        $nestedData['Hardware Standard (Teqtivity) '] = disableCSVInjection(optional($asset->makeAndModel)->name);
        $nestedData['Asset Location (Teqtivity) '] = optional(optional($asset)->location)->room_name;
        $nestedData['Asset Status (Teqtivity)'] = disableCSVInjection(optional($asset->assetStatus)->name);
        $nestedData['User (Teqtivity)'] = disableCSVInjection($asset->user ? optional($asset->user)->first_name . ' ' . optional($asset->user)->email : '');
        $nestedData['User Status (Teqtivity)'] = disableCSVInjection(optional(optional($asset)->user)->status == 1 ? 'Active' : 'Terminated');
        $nestedData['User Department (Teqtivity)'] = disableCSVInjection(optional(optional(optional($asset)->user)->department)->name);
        $nestedData['User Work Location (Teqtivity)'] = disableCSVInjection(($asset->user) ? optional($asset->user)->city : '');
        $nestedData['Host Name (Teqtivity)'] = disableCSVInjection($asset->host_name);
        $nestedData['MAC Address (Teqtivity)'] =  disableCSVInjection($asset->mac);
        $nestedData['IMEI (Teqtivity)'] = disableCSVInjection($asset->imei);
        $nestedData['Carrier (Teqtivity)'] = disableCSVInjection(optional($asset->carrier)->name);
        $nestedData['Carrier Country (Teqtivity)'] = disableCSVInjection(optional(optional($asset->carrier)->country)->name);
        $nestedData['Last Seen'] = $asset->latestAssetHistory ? $asset->latestAssetHistory->updated_at : '';

        return $nestedData;
    }
}
