<?php

namespace App\Services\DiscoveryTools\JumpCloud;

use App\Http\Responses\ReportOutputData;
use App\Repositories\DiscoveryTools\JumpCloudRepository;

use App\User;

abstract class AbstractJumpCloudService
{
    public function __construct(protected JumpCloudRepository $repository, protected ReportOutputData $reportOutputData) {}

    abstract public function data();
    abstract public function exportData();

    /**
     * Taking the input data
     * @return Array
     */
    public function getInputData()
    {
        $requestData = request('form') ?? request()->all();
        return [
            'serial_no'          => $requestData['serial_no'] ?? '',
            'asset_type'         => $requestData['asset_type'] ?? '',
            'asset_status'       => $requestData['asset_status'] ?? '',
            'manufacturer_teq'   => $requestData['manufacturer_teq'] ?? '',
            'make_and_model'     => $requestData['make_and_model'] ?? '',
            'manufacturer'       => $requestData['manufacturer'] ?? '',
            'model'              => $requestData['model'] ?? '',
            'jumpcloud_user'     => $requestData['jumpcloud_user'] ?? '',
            'teq_user'           => $requestData['teq_user'] ?? '',
            'location'           => $requestData['room_name'] ?? '',
            'user_status'        => $requestData['user_status'] ?? '',
            'user_department'    => $requestData['user_department'] ?? '',
            'work_location'      => $requestData['work_location'] ?? '',
            'last_seen_from'     => $requestData['last_seen_from'] ?? '',
            'last_seen_to'       => $requestData['last_seen_to'] ?? '',
            'platform'           => $requestData['platform'] ?? '',
            'host_name'          => $requestData['host_name'] ?? '',
            'mac_address'        => $requestData['mac_address'] ?? '',
            'carrier'            => $requestData['carrier'] ?? '',
            'carrier_country'    => $requestData['carrier_country'] ?? '',
            'imei'               => $requestData['imei'] ?? '',
        ];
    }

    /**
     * Filer the results based on filters
     * 
     * @param $assets DB Query
     * 
     * @return Query
     */
    public function filter($assets)
    {
        $inputData = $this->getInputData();

        $assets     = $assets->RelationData($inputData['serial_no'], 'jump_cloud_devices.serial_no');
        $assets     = $assets->RelationData($inputData['model'], 'jump_cloud_devices.model');
        $assets     = $assets->RelationData($inputData['manufacturer'], 'jump_cloud_devices.manufacturer');
        $assets     = $assets->RelationData($inputData['jumpcloud_user'], 'jump_cloud_devices.user_name');
        $assets     = $assets->RelationData($inputData['mac_address'], 'jump_cloud_devices.mac_address');
        $assets     = $assets->RelationData($inputData['host_name'], 'jump_cloud_devices.device_name');


        $assets     = $this->repository->filterWithAssetFields($assets, 'asset_type_id', $inputData['asset_type']);
        $assets     = $this->repository->filterWithAssetFields($assets, 'make_and_model_id', $inputData['make_and_model']);
        $assets     = $this->repository->filterWithAssetFields($assets, 'user_id', $inputData['teq_user']);
        $assets     = $this->repository->filterWithAssetFields($assets, 'location_id', $inputData['location']);
        $assets     = $this->repository->filterWithAssetFields($assets, 'asset_status_id', $inputData['asset_status']);

        $assets     = $this->repository->filterWithAssetMakeAndModelFields($assets, 'manufacturer_id', $inputData['manufacturer_teq']);

        $assets     = $this->repository->filterWithTeqtivityUserFields($assets, 'department_id', $inputData['user_department']);
        $assets     = $this->repository->filterWithTeqtivityUserFields($assets, 'status', $inputData['user_status']);
        $assets     = $this->repository->filterWithTeqtivityUserFields($assets, 'city', $inputData['work_location']);

        $assets     = $this->repository->filterWithLastSeen($assets, $inputData['last_seen_from'], $inputData['last_seen_to']);

        // $assets     = $this->repository->filterWithUserFields($assets, 'status', $inputData['user_status']);
        // $assets     = $this->repository->filterWithUserFields($assets, 'city', $inputData['work_location']);

        return $assets;
    }

    /**
     * Creating array for Data table data
     * @param $asset Query Result
     *
     * @return array
     */
    public function getCommonNestedData($asset)
    {
        $nestedData = [];

        $nestedData = array_merge_recursive(
            $nestedData,
            $this->getTeqtivityData($asset->asset),
            $this->getTeqtivityUserData($asset->asset),
            $this->getJumpcloudSystemData($asset)
        );

        return $nestedData;
    }

    /**
     * Collecting the asset data from Teqtivity
     * @param $asset
     * @return array
     */
    public function getTeqtivityData($asset)
    {
        $nestedData = [];

        $nestedData['report']['asset_type']              = $nestedData['export']['Asset Type (Teqtivity)'] = disableCSVInjection(optional(optional($asset)->assetType)->name);
        $nestedData['report']['manufacturer_teq']        = $nestedData['export']['Manufacturer (Teqtivity)'] = disableCSVInjection(optional(optional(optional($asset)->makeAndModel)->manufacturer)->name);
        $nestedData['report']['hardware_teq']            = $nestedData['export']['Hardware Standard (Teqtivity)'] = disableCSVInjection(optional(optional($asset)->makeAndModel)->name);
        $nestedData['report']['location_teq']            = $nestedData['export']['Asset Location (Teqtivity)'] = disableCSVInjection(optional(optional($asset)->location)->room_name);
        $nestedData['report']['asset_status_teq']        = $nestedData['export']['Asset Status (Teqtivity)'] = disableCSVInjection(optional(optional($asset)->assetStatus)->name);
        $nestedData['report']['last_modified_date']      = $nestedData['export']['Last Modified Date']   = optional(optional($asset)->latestAssetHistory)->updated_at;

        return $nestedData;
    }

    /**
     * Collecting the user data from Teqtivity
     * @param $asset
     * @return array
     */
    public function getTeqtivityUserData($asset)
    {
        $nestedData = [];

        $nestedData['report']['teq_user']                = $nestedData['export']['User (Teqtivity)'] = disableCSVInjection(optional(optional($asset)->user)->email);
        $nestedData['report']['teq_user_status']         = $nestedData['export']['User Status (Teqtivity)'] = isset(optional($asset)->user) ? (disableCSVInjection(optional(optional($asset)->user)->status == 1 ? 'Active' : 'Terminated')) : '';
        $nestedData['report']['teq_user_department']     = $nestedData['export']['User Department (Teqtivity)'] = disableCSVInjection(optional(optional(optional($asset)->user)->department)->name);
        $nestedData['report']['teq_user_work_location']  = $nestedData['export']['User Work Location (Teqtivity)'] = disableCSVInjection(optional(optional($asset)->user)->city);

        return $nestedData;
    }

    /**
     * Creating array for export data
     * @param $asset Query Result
     *
     * @return array
     */
    public function getJumpcloudSystemData($asset)
    {
        $nestedData = [];

        $nestedData['report']['serial_no']               = $nestedData['export']['Serial # (JumpCloud)'] = disableCSVInjection($asset->serial_no);
        $nestedData['report']['jumpcloud_user']          = $nestedData['export']['User (JumpCloud)'] = disableCSVInjection($asset->user_name);
        $nestedData['report']['hardware']                = $nestedData['export']['Hardware Standard (JumpCloud)']     = disableCSVInjection($asset->model);
        $nestedData['report']['manufacturer']            = $nestedData['export']['Manufacturer (JumpCloud)'] = disableCSVInjection($asset->manufacturer);
        $nestedData['report']['host_name']               = $nestedData['export']['Host Name (JumpCloud)'] = disableCSVInjection($asset->host_name);
        $nestedData['report']['mac_address']             = $nestedData['export']['MAC Address (JumpCloud)'] = disableCSVInjection($asset->mac_address);
        $nestedData['report']['imei']                    = $nestedData['export']['IMEI (JumpCloud)'] = disableCSVInjection($asset->imei);
        $nestedData['report']['last_seen']               = $nestedData['export']['Last Seen Date'] = disableCSVInjection($asset->last_seen ?? '');
        $nestedData['report']['platform']                = $nestedData['export']['OS Platform'] = disableCSVInjection($asset->platform);

        return $nestedData;
    }

    /**
     * Generating data table data from qurey result
     * @param $users Query Result
     * @param $start Starting index
     * @param $data Collection valriable that will return
     *
     * @return array
     */
    public function getReportData($users, $start, $data)
    {
        $parentIndex = $start;

        foreach ($users as $user) {
            $parentIndex++;

            $nestedData = $this->getNestedData($user, $parentIndex);
            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Creating array for Data table data
     * @param $asset Query Result
     * @param $index Starting index
     * @param $email route for user details
     *
     * @return array
     */
    public function getNestedData($jumbcloudAsset, $index)
    {
        $assetLink  = $jumbcloudAsset->asset ? route('assets.show', $jumbcloudAsset->asset->id) : 'javascript:void(0)';
        $teqtivityUser = optional($jumbcloudAsset->asset)->user;
        $teqtivityUserLink  = $teqtivityUser ? route('users.show', $teqtivityUser->id) : 'javascript:void(0)';
        $serialNo  = $jumbcloudAsset->asset ? ("<a href=$assetLink>" . $jumbcloudAsset->serial_no . "</a>") : $jumbcloudAsset->serial_no;

        $jumpcloudUserLink   = $jumbcloudAsset->user_id ? route('users.show', $jumbcloudAsset->user_id) : 'javascript:void(0)';
        $jumpcloudUser = $jumbcloudAsset->user ? ("<a href=$jumpcloudUserLink>" . $jumbcloudAsset->user_name . "</a>") : $jumbcloudAsset->user_name;

        $nestedData = $this->getCommonNestedData($jumbcloudAsset)['report'];

        $nestedData['id']  = $index;
        $nestedData['serial_no'] = $serialNo;
        $nestedData['teq_user']   = $teqtivityUser ? ("<a href=$teqtivityUserLink>" . $teqtivityUser->email . "</a>") : '';
        $nestedData['jumpcloud_user']  = $jumpcloudUser;
        $nestedData['create_asset'] = $this->getCreateAssetButton($jumbcloudAsset);
        $nestedData['update_asset'] = $this->getUpdateDiscoveryAssetButton($jumbcloudAsset->asset, optional($jumbcloudAsset->asset)->assetStatus);

        return $nestedData;
    }

    /**
     * Generating data to export
     * @param $assets Query Result
     * @param $start Starting index
     * @param $data Collection valriable that will return
     * @param $requiredColumns colums to be shown in export file

     * @return array
     */
    public function getReportExportData($assets, $data, $requiredColumns)
    {
        foreach ($assets as  $asset) {
            $nestedData = $this->getExportNestedData($asset, $requiredColumns);
            $data[] = $nestedData;
        }
        return $data;
    }

    /**
     * Creating array for export data
     * @param $asset Query Result
     * @param $requiredColumns columns to be added in export file
     *
     * @return array
     */
    public function getExportNestedData($asset, $requiredColumns)
    {
        setUnlimitedExecutionTimeAndMemoryLimit();
        $data = [];

        $nestedData = $this->getCommonNestedData($asset)['export'];

        foreach ($requiredColumns as $column) {
            $data[$column] =  $nestedData[$column];
        }

        return $data;
    }

    /**
     * Set up Create Asset Button
     * @param mixed $asset
     */
    public function getCreateAssetButton($asset)
    {
        if (auth()->user()->can('IT Assets Edit') && $asset) {
            return '<a href="javascript:;" data-asset="' . $asset->id . '" data-asset-serial="' . $asset->serial_no . '" class="btn btn-primary btn-sm create-asset">Create Asset</a>';
        }
        return '';
    }

    /**
     * Retrieves the update discovery asset button.
     *
     * @param mixed $asset The asset object.
     * @param mixed $assetStatus The asset status object.
     * @return string The HTML code for the update asset button.
     */
    public function getUpdateDiscoveryAssetButton($asset, $assetStatus)
    {
        if (auth()->user()->can('IT Assets Edit') && $asset) {
            return '<a href="javascript:;" data-asset="' . $asset->id . '" data-asset-serial="' . $asset->serial_no . '" data-asset-current-status="' . optional($assetStatus)->slug . '" data-parent-asset="' . $asset->parent_asset_id . '" class="btn btn-primary btn-sm update-discrepancy-asset">Update Asset</a>';
        }
        return '';
    }
}
