<?php

namespace App\Services\DiscoveryTools\Jamf;

use App\Http\Traits\AutoCreateAssignTrait;
use App\Models\ApiCredential;
use App\Models\Asset;
use App\Models\AssetStatus;
use App\Models\AssetType;
use App\Models\JamfComputer;
use App\Models\Location;
use App\User;
use Illuminate\Support\Facades\Cache;

class JamfMobileDataSyncService
{

    use AutoCreateAssignTrait;

    protected $installUserName;

    public function __construct(protected JamfLockManagementService $lockManagementService)
    {
        $this->installUserName = 'team-it-devices@goforward.com';
    }

    /**
     * Update the Jamf records based on the API response
     *
     * @param $item API response
     * @param $hardwaredata  Hardwaredetails API reponse
     * @param $$deviceDetails  Device Details API reponse
     */
    public function updateJamfData($item, $deviceDetails)
    {
        $data = $this->getData($item, $deviceDetails);

        if (strlen($data['serial_no']) <= 0) {
            return 0;
        }
        $assetTypeId   = Cache::remember('tablet_asset_type_id', 30, function () {
            return AssetType::firstOrCreate(['name' => 'Tablet', 'slug' => 'tablet'])->id;
        });


        $assetStatusId = Cache::remember('assigned_asset_status_id', 30, function () {
            return AssetStatus::firstOrCreate(['name' => 'Assigned', 'slug' => 'assigned'])->id;
        });

        $jamfDataId = JamfComputer::updateOrCreate(['serial_no' => $data['serial_no']], $data)->id;

        $data['poNumber']      = $deviceDetails->ios->purchasing->poNumber;
        $data['asset_type_id'] = $assetTypeId;
        $data['asset_tag']     = $deviceDetails->assetTag;

        //Auto Create and Assign Asset
        if (($data['asset_id'] == 0) && ($data['user_id'] != 0)) {
            $techSpec = $this->getTechSpec($data, 'jamf_model');

            $data['asset_status_id'] = $assetStatusId;
            $data['jamf_id']         = $jamfDataId;
            $this->autoCreateAndAssign($data, $techSpec, "Jamf");
        }

        return 1;
    }

    /**
     * Making array with needed data for saving to DB
     * @param $item CSV data array
     * @param $deviceDetails Device details array
     * @return array
     */
    public function getData($item, $deviceDetails)
    {
        $userId = 0;
        $assetId = 0;

        if ($deviceDetails->location->emailAddress) {
            $user = User::select('id')->where('email', $deviceDetails->location->emailAddress)->first();
            $userId = $user ? $user->id : 0;
        }

        if ($deviceDetails->serialNumber) {
            $asset = Asset::select('id')->where('serial_no', $deviceDetails->serialNumber)->first();
            $assetId = $asset ? $asset->id : 0;
        }


        $data['user_id'] = $userId;
        $data['asset_id'] = $assetId;
        $data['checkin_date'] = $deviceDetails->lastInventoryUpdateTimestamp;
        $data['user_name'] = $deviceDetails->location->username ?? '';
        $data['jamf_email'] = $deviceDetails->location->emailAddress;
        $data['name'] = $deviceDetails->name;
        $data['serial_no'] = $deviceDetails->serialNumber;
        $data['model'] = $item->model;
        $data['manufacturer'] = $deviceDetails->type == 'ios' ? 'Apple' : 'Samsung';
        $data['os_version'] = $deviceDetails->osVersion;
        $data['os_build'] = $deviceDetails->osBuild;
        $data['mac_address'] = $deviceDetails->wifiMacAddress;
        $data['udid'] = $deviceDetails->udid;
        $data['ip_address'] = $deviceDetails->ipAddress;
        $data['hard_drive'] = null;
        $data['disk_capacity'] = convertSpecsToReadableValue($deviceDetails->ios->capacityMb, 'disk_capacity');
        $data['ram'] = null;
        $data['processor'] = null;
        $data['processor_speed'] = null;
        $data['processor_type']  =  null;
        $data['jamf_device_id']  =  $item->id ?? null;
        $data['api_type'] = 'Mobile';
        $data['activation_lock']  = optional($deviceDetails->ios->security)->activationLockEnabled;

        return $data;
    }

    /**
     * GET Techspec string for finding tech specs
     * @param mixed $jamfData
     *
     * @return [type]
     */
    public function getTechSpecString($jamfData)
    {
        return $jamfData['disk_capacity'];
    }

    /**
     * Update Dicovery tool table asset ID
     * @param mixed $jamfData
     * @param mixed $assetId
     *
     * @return [type]
     */
    public function updateDiscoveryAssetId($jamfData, $assetId)
    {
        return JamfComputer::find($jamfData['jamf_id'])->update(['asset_id' => $assetId]);
    }

    /**
     * Cretes the devices url to link from asset detail page
     * @param mixed $item
     *
     * @return [type]
     */
    private function createDeviceUrl($item)
    {
        if (!isset($item->id)) {
            return '';
        }
        $jamfCredentials = ApiCredential::where('slug', 'jamf')->first();
        $baseUrl = optional($jamfCredentials)->url;
        return $baseUrl . 'computers.html?id=' . $item->id;
    }
}
