<?php

namespace App\Services\DiscoveryTools\Jamf;

use App\Events\BulkUpdates;
use App\Models\Asset;
use App\Models\AssetLockDetails;
use App\Models\JamfComputer;
use App\Services\Integrations\JamfIntegration;
use App\Services\Integrations\Tickets\TicketManagementService;
use App\User;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

/**
 * Class used for managing Jamf devices lock/unlock. 
 */
class JamfLockManagementService
{

    public function __construct(protected TicketManagementService $ticketManagementService, protected JamfIntegration $jamfApiService) {}

    /** Lock the device
     * 
     * @return [type]
     */
    public function lockDevice()
    {
        if (app()->isProduction() === false) {
            return true;
        }

        $asset = Asset::where('id', request('asset_id'))->first();
        $lockCode = request('lock_code') ?? date('dmy'); // default lock code should be the current date in format DDMMYY

        $asset->update(['lock_code' => $lockCode]); // Storing lock code in assets tabel before updating on MDM.

        try {
            $deviceDetail = JamfComputer::where('serial_no', $asset->serial_no)->first();
            if (!$deviceDetail) {
                return false;
            }

            $computerId = $deviceDetail->jamf_device_id;
            $serialNo = $deviceDetail->serial_no;

            $response = $this->jamfApiService->lockComputer($computerId, $lockCode);
            if (!$response) {
                return false;
            }

            $xmlResponse = $response->getBody()->getContents();

            Log::channel('daily')->info("Jamf device lock response of asset-$serialNo :\n" . $xmlResponse);
            $statusCode = $response->getStatusCode();
            if ($statusCode !== 201) {
                return false;
            }

            $lockStatus = $this->storeLockDetails($deviceDetail->asset_id, $lockCode, $xmlResponse);

            /* $dummyResponse = '<?xml version="1.0" encoding="UTF-8"?><computer_command><command><name>DeviceLock</name><command_uuid>856e0de7-24f1-42c5-b253-4835b242989d</command_uuid><computer_id>1305</computer_id></command></computer_command>';*/

            // $lockStatus = $this->storeLockDetails($deviceDetail->asset_id, $lockCode, $dummyResponse);

            $this->createDeviceHistory($deviceDetail, $lockStatus);

            $this->createTicketHistory($deviceDetail);

            return true;
        } catch (Exception $e) {
            // dd($e->getMessage(), $e->getLine(), $e->getFile());
            Log::channel('daily')->error("Failed to lock asset $serialNo:-" . $e->getMessage());

            return false;
        }

        return true;
    }

    /**
     * store LockDetails
     *
     * @param  int $assetId
     * @param  string $lockCode
     * @param  mixed $response
     * @return void
     */
    private function storeLockDetails($assetId, $lockCode, $response)
    {
        Log::channel('daily')->info('from storelockDetails ' . $response);
        $responseJson = convertXmlToJson($response);
        Log::channel('daily')->info('from storelock after convertXmlToJson ' . json_encode($responseJson));
        $responseUuid = optional($responseJson->command)->command_uuid ?? null;

        $lockStatus = 'Pending';

        if ($responseUuid) {
            $lockDetails = $this->jamfApiService->getCommandDetailsByUuid($responseUuid);

            if ($lockDetails) {
                Log::channel('daily')->info(json_encode($lockDetails));
                $apnsResultStatus = optional(optional($lockDetails->computer_command)->general)->apns_result_status ?? null;
                $lockStatus = ($apnsResultStatus == "Acknowledged") ? 'Lock Confirmed' : 'Pending';
                $dateSent = optional(optional($lockDetails->computer_command)->general)->date_sent ?? Carbon::now()->format('Y-m-d H:i:s');
            }
        }

        $assetLockDetails = AssetLockDetails::create([
            'uuid' => $responseUuid,
            'asset_id' => $assetId,
            'unlock_code' => $lockCode,
            'lock_status' => $lockStatus ?? '',
            'lock_note' => request('lock_notes'),
            'updated_at' => $dateSent,
        ]);

        return $assetLockDetails;
    }

    /** Create Link Info to Ticket Service
     * @param mixed $deviceDetail
     * 
     * @return [type]
     */
    private function createTicketHistory($deviceDetail)
    {
        $user = User::find(Auth::id());
        $ticketDescription =  __('jira.LockDevice', [
            'assetname' => $deviceDetail->serial_no,
            'assetid' => $deviceDetail->asset_id,
            'username' => $user ? $user->first_name . ' ' . $user->last_name : '',
            'useremail' => $user->email ?? '',
            'device_action' => 'Locked',
        ]);

        $this->ticketManagementService->addComment(Auth::id(), request('ticket_no'), $ticketDescription, "Device Locked");

        return true;
    }

    /**
     * It creates a history record for device lock and adds comment to ticket.
     *
     * @param JamfComputer $deviceDetail This is the device object that you're locking.
     * @param mixed $lockDetails
     */
    private function createDeviceHistory($deviceDetail, $lockDetails)
    {
        $user = User::find(Auth::id());
        $assetHistorydescription = __('history.LockDevice', [
            'assetname'  => $deviceDetail->serial_no,
            'assetId'    => $deviceDetail->asset_id,
            'device_action' => 'Locked'
        ]);

        $assetHistory = [
            'asset_id' => $deviceDetail->asset_id,
            'action' => 'lock_device',
            'ticket_no' => request('ticket_no'),
            'ticket_service_provider' => config('ticket-integration.service'),
            'user_id' => $user->id ?? '',
            'old_lock_code' => optional($lockDetails->lockHistory)->new_lock_code ?? '',
            'new_lock_code' => request('lock_code'),
            'lock_note' => request('lock_notes'),
            'new_lock_status' => $lockDetails->lock_status,
            'description' => $assetHistorydescription,
        ];

        event(new BulkUpdates($assetHistory));
        return true;
    }

    /**
     * Updates AssetLockDetails if device is unlocked. 
     * Checks if jamf checkin_date is greater than lock updated date to confirm whether device is unlocked.
     *
     * @param array $jamfData 
     * @return boolean
     */
    public function updateAssetLockDetails($jamfData)
    {
        $checkinDate = Carbon::parse($jamfData['checkin_date']);
        if (!$checkinDate || !$checkinDate->isValid()) {
            return false;
        }

        $latestAssetLock = AssetLockDetails::whereLockStatus('Lock Confirmed')->where('asset_id', $jamfData['asset_id'])->latest()->first();

        // Device is unlocked
        if ($latestAssetLock && Carbon::parse($latestAssetLock->updated_at)->lt($checkinDate)) {
            AssetLockDetails::whereLockStatus('Lock Confirmed')->where('asset_id', $jamfData['asset_id'])->delete();
            Asset::where('id', $jamfData['asset_id'])->update(['lock_code' => NULL]);
        }

        return true;
    }
}
