<?php

namespace App\Services\DiscoveryTools\Jamf;

use App\Events\BulkUpdates;
use App\Http\Traits\AutoCreateAssignTrait;
use App\Models\Asset;
use App\Models\AssetStatus;
use App\Models\AssetType;
use App\Models\JamfComputer;
use App\Models\Location;
use App\Repositories\DiscoveryTools\JamfRepository;
use App\Services\Integrations\JamfIntegration;
use Facades\App\Services\Asset\AssetStatusService;
use App\User;
use Cache;
use Carbon\Carbon;
use Exception;

class JamfDataSyncService
{

    use AutoCreateAssignTrait;
    public function __construct(protected JamfRepository $repository, protected JamfIntegration $jamfApiService) {}

    /**
     * Update the Jamf records based on the API response
     * 
     * @param $item API response
     * @param $hardwaredata  Hardwaredetails API reponse
     * @param $$deviceDetails  Device Details API reponse
     */
    public function updateJamfData($item, $deviceDetails)
    {
        $data = $this->getData($item, $deviceDetails);
        if (strlen($data['serial_no']) > 0) {

            $assetTypeId = Cache::remember('asset_type_computer', 3600, function () {
                return AssetType::firstOrCreate(['name' => 'Computer', 'slug' => 'computer'])->id;
            });

            $assetStatusId = Cache::remember('asset_status_assigned', 3600, function () {
                return AssetStatus::firstOrCreate(['name' => 'Assigned', 'slug' => 'assigned'])->id;
            });

            $jamfDataId = JamfComputer::updateOrCreate(['serial_no' => $data['serial_no']], $data)->id;

            $data['make']          = $deviceDetails->hardware->make ?? null;
            $data['poNumber']      = $deviceDetails->purchasing->poNumber ?? null;
            $data['asset_type_id'] = $assetTypeId;
            $data['asset_tag']      = $item->general->assetTag ?? null;


            //Auto Create and Assign Asset
            if (($data['asset_id'] == 0) && ($data['user_id'] != 0)) {
                $techSpec = $this->getTechSpec($data, 'jamf_model');

                $data['asset_status_id'] = $assetStatusId;
                $data['jamf_id']         = $jamfDataId;
                $this->autoCreateAndAssign($data, $techSpec, "Jamf");
            }
        }
    }

    /**
     * Retrieves data for a given item and device details.
     *
     * @param mixed $item The item to retrieve data for.
     * @param mixed $deviceDetails The device details object.
     * @return array The retrieved data.
     */
    public function getData($item, $deviceDetails)
    {
        $userId = 0;
        $assetId = 0;

        if ($deviceDetails->userAndLocation->email) {
            $user = User::select('id')->where('email', $deviceDetails->userAndLocation->email)->first();
            $userId = $user ? $user->id : 0;
        }

        if ($deviceDetails->hardware->serialNumber) {
            $asset = Asset::select('id')->where('serial_no', $deviceDetails->hardware->serialNumber)->first();
            $assetId = $asset ? $asset->id : 0;
        }

        $data['user_id']         = $userId;
        $data['asset_id']        = $assetId;
        $data['checkin_date']    = $item->general->lastContactTime ?? null;
        $data['jamf_email']      = $deviceDetails->userAndLocation->email ?? null;
        $data['name']            = $item->general->name ?? null;
        $data['serial_no']       = $deviceDetails->hardware->serialNumber ?? null;
        $data['model']           = $deviceDetails->hardware->model ?? null;
        $data['manufacturer']    = optional($deviceDetails->hardware)->make ?? null;
        $data['os_version']      = $deviceDetails->operatingSystem->version ?? null;
        $data['os_buid']         = $deviceDetails->operatingSystem->build ?? null;
        $data['mac_address']     = $deviceDetails->hardware->macAddress ?? null;
        $data['udid']            = $item->udid;
        $data['ram']             = isset($deviceDetails->hardware->totalRamMegabytes) ? convertSpecsToReadableValue($deviceDetails->hardware->totalRamMegabytes, 'ram') : null;
        $data['hard_drive']      = $this->getHardDriveAttributes($deviceDetails, 'model');
        $data['disk_capacity']   = convertSpecsToReadableValue($this->getHardDriveAttributes($deviceDetails, 'sizeMegabytes'), 'disk_capacity');
        $data['processor']       = $deviceDetails->hardware->processorType ?? null;
        $data['processor_speed'] = isset($deviceDetails->hardware->processorSpeedMhz) ? convertSpecsToReadableValue($deviceDetails->hardware->processorSpeedMhz, 'processor_speed') : null;
        $data['processor_type']  = $deviceDetails->hardware->processorArchitecture ?? null;
        $data['jamf_device_id']  =  $item->id ?? null;
        $data['api_type']        = 'Computer';
        $data['file_vault_status'] = $this->getFileVaultStatus($deviceDetails);
        $data['activation_lock']  = optional($deviceDetails->security)->activationLockEnabled;
        $data['recovery_lock_password']  = $this->getRecoveryLockPassword($deviceDetails->id ?? 0);
        $data['file_vault_recovery_key']  = $this->getFileVaultRecoveryKey($deviceDetails->id ?? 0);
        $data['activation_lock_bypass_code']  = null;

        return $data;
    }

    /**
     * Format technical specifications in a string format
     *
     * @param array $jamfData
     * @return string
     */
    public function getTechSpecString($jamfData)
    {
        if (!$jamfData['processor_speed'] && !$jamfData['ram'] && !$jamfData['disk_capacity']) {
            return 'Ghz/GB/GB';
        }

        return (($jamfData['processor_speed'] && $jamfData['processor_speed'] != '') ? $jamfData['processor_speed'] : 'Ghz') . " / " . (($jamfData['ram'] && $jamfData['ram'] != '-') ? $jamfData['ram'] : 'GB') . " / " . (($jamfData['disk_capacity'] && $jamfData['disk_capacity'] != '-') ? $jamfData['disk_capacity'] : 'GB');
    }

    /**
     * Update Dicovery tool table asset ID
     * @param mixed $jamfData
     * @param mixed $assetId
     * 
     * @return [type]
     */
    public function updateDiscoveryAssetId($jamfData, $assetId)
    {
        return JamfComputer::find($jamfData['jamf_id'])->update(['asset_id' => $assetId]);
    }

    /**
     * Auto re assigns the discreapancy assets.
     * @return [type]
     */
    public function autoReassignAssets()
    {
        $discrepancyDevices = $this->repository->getUserMismatchDevices()->has('user')->get();
        $this->reAssignDiscrepancyAssets($discrepancyDevices, 'Jamf');
        return true;
    }

    /* Get HDD related attributes
     * 
     * @param Array $hardwareData. It is Hardware details API response
     * @param string which attribute needs to be returned
     * 
     * @return string
     */
    public function getHardDriveAttributes($hardwareData, $attribute)
    {
        //For some devices, there are multiple indexes for the disc and only one index have the proper data
        //So here we finding that proper index and taking the attributes from that index
        //Based on my research i found device=disk0 contained index have proper value
        $discArray = $hardwareData->storage->disks;
        $key = array_search('disk0', array_column($discArray, 'device'));
        $attributeValue = null;
        if ($key !== false) {
            $attributeValue = $discArray[$key]->$attribute;
        }
        return $attributeValue;
    }

    /**
     * Create prestage Device
     */
    public function createPrestageDevice($serialNumber)
    {
        if (!$serialNumber) {
            return true;
        }

        $asset = Asset::select('id')->where('serial_no', $serialNumber)->first();
        $assetId = $asset ? $asset->id : 0;

        $jamfAsset = JamfComputer::select('id', 'api_type', 'asset_id')->where(['serial_no' => $serialNumber])->first();

        // if jamfAsset not found then create
        if (empty($jamfAsset)) {
            $jamfDataId = JamfComputer::Create(['serial_no' => $serialNumber, 'api_type' => 'prestage', 'asset_id' => $assetId])->id;
            //If asset ID is empty, we will create prestage device with just serial no
            $this->autoCreatePrestageTeqtivityAsset($assetId, $jamfDataId, $serialNumber);
        }

        // if jamfAsset found and api_type is prestage then update
        if ($jamfAsset->api_type == 'prestage' && $jamfAsset->asset_id != $assetId) {
            $jamfAsset->update(['asset_id' => $assetId]);
        }

        return true;
    }

    /**
     * Creates a prestage teqtivity asset.
     *
     * @param int $assetId The asset ID.
     * @param int $jamfDataId The Jamf Data ID.
     * @param string $serialNumber The serial #.
     * @return bool
     */
    private function autoCreatePrestageTeqtivityAsset($assetId, $jamfDataId, $serialNumber)
    {
        if ($assetId == 0) {
            // Here just defining a default location 
            $location = Location::select('id', 'room_name')->where('room_name', 'Storage Shed')->first();
            $status = AssetStatus::where('slug', AssetStatusService::whenCreatingAsset())->first();

            $assetData['location_id']       = optional($location)->id;
            $assetData['asset_status_id']   = optional($status)->id;
            $assetData['serial_no']         = $serialNumber;

            $asset = Asset::create($assetData);
            $this->createPrestageDeviceCreatedHistory($asset, $location, $status);

            //update the asset id in the discovery tool table
            $assetData['jamf_id'] = $jamfDataId;
            $this->updateDiscoveryAssetId($assetData, $asset->id);
        }

        return true;
    }

    /**
     * Creating array for update the asset history for Received
     *
     * @param Asset $asset
     * @param Location $location
     * @param Status $status
     * @return Array
     */
    public function createPrestageDeviceCreatedHistory($asset, $location, $status)
    {
        $description = __('history.Created', [
            'assetname'     => $asset->serial_no,
            'assetid'       => $asset->id,
            'newroomname'   => optional($location)->room_name,
        ]);

        $assetHistory = [
            'asset_id'          => $asset->id,
            'new_location_id'   => optional($location)->id,
            'new_asset_status_id' => $status->id,
            'new_value'         => $asset->serial_no,
            'action'            => 'created',
            'description'       => $description,
            'created_by'        => 'Jamf',
            'created_at'        => Carbon::now()->format('Y-m-d H:i:s'),
            'updated_at'        => Carbon::now()->format('Y-m-d H:i:s'),
        ];
        event(new BulkUpdates($assetHistory));
    }

    /**
     * Get the FileVault Attribute from the API response
     * 
     * @param Array Hardware details API response
     */
    public function getFileVaultStatus($hardwareData)
    {
        $partition = $this->getHardDriveAttributes($hardwareData, 'partitions');
        if (!is_array($partition)) {
            return null;
        }

        $bootPartitionKey = array_search('BOOT', array_column($partition, 'partitionType'));
        $bootPartitionStatus = ($bootPartitionKey !== false) ? $partition[$bootPartitionKey]->fileVault2State : '';

        $dataPartitionKey = array_search('Data', array_column($partition, 'name'));
        $dataPartitionStatus = ($dataPartitionKey !== false) ? $partition[$dataPartitionKey]->fileVault2State : '';

        if ($dataPartitionStatus == 'ENCRYPTED' && $bootPartitionStatus == 'ENCRYPTED') {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Taking the Recovery Lock password by using device ID
     * 
     * @param $deviceId
     */
    public function getRecoveryLockPassword($deviceId)
    {
        try {
            $response = $this->jamfApiService->getRecoveryLockPasswordById($deviceId);
            return $response->recoveryLockPassword ?? null;
        } catch (Exception $e) {
            // echo $e->getMessage().' | '.$deviceId.PHP_EOL;
        }

        return null;
    }

    /**
     * Taking the File Vault Recovery Key
     */
    public function getFileVaultRecoveryKey($deviceId)
    {
        try {
            $response = $this->jamfApiService->getFileVaultInfoById($deviceId);
            return $response->personalRecoveryKey ?? null;
        } catch (Exception $e) {
            // echo $e->getMessage().' | '.$deviceId.PHP_EOL;
        }

        return null;
    }
}
