<?php

namespace App\Services\DiscoveryTools\Intune;

use App\Repositories\DiscoveryTools\IntuneRepository;
use App\User;

class NotCheckingInNotManaged extends AbstractIntuneService
{

    public function __construct(protected IntuneRepository $intuneRepository)
    {
    }

    /**
     * Making query for the report
     */
    public function data()
    {
        $inputData  = $this->getInputData();
        $assets     = $this->intuneRepository->getNotManaged();
        $assets     = $this->filterDevicesNotManaged($assets, $inputData);
        $count      = $assets->count();
        $assets = $this->getOutputData($assets, 'created_at');
        return compact('assets', 'count');
    }

    /**
     * Making the query for export
     */
    public function exportData()
    {
        $inputData  = $this->getExportInputData();
        $assets     = $this->intuneRepository->getNotManaged();
        $assets     = $this->filterDevicesNotManaged($assets, $inputData);
        $assets     = $assets->orderBy('created_at', 'desc');
        return $assets;
    }

    /**
     * Filer the results based on filters
     * 
     * @param $assets DB Query
     * @param Array $inputData   Filter data
     * 
     * @return Query
     */

    public function filterDevicesNotManaged($assets, $inputData)
    {
        $assets     = $this->intuneRepository->filterByMakeAndModelFields($assets, 'manufacturer_id', $inputData['manufacturer_id']);

        $assets     = $this->intuneRepository->filterByAssetFields($assets, 'serial_no', $inputData['serial_no']);
        $assets     = $this->intuneRepository->filterByAssetFields($assets, 'user_id', $inputData['user_id']);
        $assets     = $this->intuneRepository->filterByAssetFields($assets, 'asset_type_id', $inputData['asset_type_id']);
        $assets     = $this->intuneRepository->filterByAssetFields($assets, 'make_and_model_id', $inputData['make_and_model_id']);
        $assets     = $this->intuneRepository->filterByAssetFields($assets, 'location_id', $inputData['location_id']);
        $assets     = $this->intuneRepository->filterByAssetFields($assets, 'asset_status_id', $inputData['asset_status_id']);


        $assets     = $this->intuneRepository->filterByAssetFields($assets, 'host_name', $inputData['host_name']);
        $assets     = $this->intuneRepository->filterByAssetFields($assets, 'mac', $inputData['mac_address']);
        $assets     = $this->intuneRepository->filterByAssetFields($assets, 'imei', $inputData['imei']);
        $assets     = $this->intuneRepository->filterByAssetFields($assets, 'carrier_id', $inputData['carrier']);

        $assets     = $this->intuneRepository->filterWithCarrierCountry($assets, $inputData['carrier_country'], 'carrier');

        $assets     = $this->intuneRepository->filterWithUserFields($assets, 'status', $inputData['user_status']);
        $assets     = $this->intuneRepository->filterWithUserFields($assets, 'department_id', $inputData['user_department']);
        $assets     = $this->intuneRepository->filterWithUserFields($assets, 'city', $inputData['work_location']);


        $assets     = $this->intuneRepository->filterWithLastModifiedDate($assets, $inputData['checkin_date_from'], $inputData['checkin_date_to']);


        return $assets;
    }

    /**
     * making array for display in table
     * @param mixed $asset
     * @param mixed $index
     * @param mixed $userLink
     * 
     * @return [type]
     */
    public function getNestedData($asset, $index, $userLink)
    {

        $assetLink  = $asset ? route('assets.show', $asset->id) : 'javascript:void(0)';
        $teqtivityUser      = $asset->user;
        $teqtivityUserLink  = $teqtivityUser ? route('users.show', optional($teqtivityUser)->id ?: 0) : 'javascript:void(0)';
        $serialNo  = $asset ? ("<a href=$assetLink>" . $asset->serial_no . "</a>") : $asset->serial_no;

        $nestedData['id']               = $index;
        $nestedData['serial_no']        = $serialNo;
        $nestedData['asset_type_teqtivity']     = optional(optional($asset)->assetType)->name;
        $nestedData['manufacturer_teqtivity']     = optional(optional(optional($asset)->makeAndModel)->manufacturer)->name;
        $nestedData['hardware_teqtivity']            = optional(optional($asset)->makeAndModel)->name;
        $nestedData['location_teqtivity']     = optional(optional($asset)->location)->room_name;
        $nestedData['asset_status_teqtivity']     = $asset ? optional($asset->assetStatus)->name : '';

        $nestedData['teqtivity_user'] = $teqtivityUser ? $teqtivityUser->first_name . ' ' . $teqtivityUser->last_name : '';
        $nestedData['teqtivity_user']   = $teqtivityUser ? ("<a href=$teqtivityUserLink>" . $teqtivityUser->first_name . ' - ' . $teqtivityUser->email . "</a>") : '';
        $nestedData['teqtivity_user_status']  = optional(optional($asset)->user)->status == 1 ? 'Active' : 'Terminated';
        $nestedData['teqtivity_user_department']  = optional(optional(optional($asset)->user)->department)->name;
        $nestedData['teqtivity_user_work_location']  = optional(optional($asset)->user)->city;

        $nestedData['host_name_teqtivity']        = $asset->host_name;
        $nestedData['mac_address_teqtivity']      = $asset->mac;
        $nestedData['imei_teqtivity']             = $asset->imei;

        $nestedData['carrier']      = optional($asset->carrier)->name;
        $nestedData['carrier_country']  = optional(optional($asset->mobileCarrier)->country)->name;
        $nestedData['last_modified_date']   = optional($asset)->latestAssetHistory ? optional(optional($asset)->latestAssetHistory)->updated_at : '';

        return $nestedData;
    }


    /**
     * Creating array for export data
     * @param $asset Query Result
     * @param $requiredColumns columns to be added in export file
     *
     * @return array
     */
    public function getExportNestedData($asset, $requiredColumns = null)
    {

        $nestedData['Serial # (Teqtivity)'] = disableCSVInjection($asset->serial_no);
        $nestedData['Asset Type (Teqtivity)'] = optional(optional($asset)->assetType)->name;
        $nestedData['Manufacturer (Teqtivity)'] = optional(optional(optional($asset)->makeAndModel)->manufacturer)->name;
        $nestedData['Hardware Standard (Teqtivity) '] = disableCSVInjection(optional($asset->makeAndModel)->name);
        $nestedData['Asset Location (Teqtivity) '] = optional(optional($asset)->location)->room_name;
        $nestedData['Asset Status (Teqtivity)'] = disableCSVInjection(optional($asset->assetStatus)->name);
        $nestedData['User (Teqtivity)'] = disableCSVInjection($asset->user ? optional($asset->user)->first_name . ' ' . optional($asset->user)->email : '');
        $nestedData['User Status (Teqtivity)'] = disableCSVInjection(optional(optional($asset)->user)->status == 1 ? 'Active' : 'Terminated');
        $nestedData['User Department (Teqtivity)'] = disableCSVInjection(optional(optional(optional($asset)->user)->department)->name);
        $nestedData['User Work Location (Teqtivity)'] = disableCSVInjection(($asset->user) ? optional($asset->user)->city : '');
        $nestedData['Host Name (Teqtivity)'] = disableCSVInjection($asset->host_name);
        $nestedData['MAC Address (Teqtivity)'] =  disableCSVInjection($asset->mac);
        $nestedData['IMEI (Teqtivity)'] = disableCSVInjection($asset->imei);
        $nestedData['Carrier (Teqtivity)'] = disableCSVInjection(optional($asset->carrier)->name);
        $nestedData['Carrier Country (Teqtivity)'] = disableCSVInjection(optional(optional($asset->carrier)->country)->name);
        $nestedData['Last Seen'] = $asset->latestAssetHistory ? $asset->latestAssetHistory->updated_at : '';

        return $nestedData;
    }
}
