<?php

namespace App\Services\DiscoveryTools\Intune;

use App\Http\Traits\AutoCreateAssignTrait;
use App\Http\Traits\AutoHardwareUpdateTrait;
use App\Models\Asset;
use App\Models\AssetStatus;
use App\Models\AssetType;
use App\Models\IntuneComputer;
use App\Repositories\DiscoveryTools\IntuneRepository;
use App\User;


use Cache;
use Exception;
use Illuminate\Support\Facades\Log;

class IntuneSyncService
{
    use AutoCreateAssignTrait, AutoHardwareUpdateTrait;

    public function __construct(protected IntuneRepository $repository) {}

    // /**
    //  * Construct attributes for intune update and save
    //  */
    public function updateIntuneSpecs($data)
    {
        return $intuneId = IntuneComputer::updateOrCreate(['serial_no' => $data['serial_no']], $data)->id;
    }


    /**
     * Making array with needed data for saving to DB
     * @param $item CSV data array
     * @return array
     */
    public function getData($item)
    {
        $userId = 0;
        $assetId = 0;
        if ($item['emailAddress']) {
            $user = User::where('email', $item['emailAddress'])->first();
            $userId = $user ? $user->id : 0;
        }

        if ($item['serialNumber']) {
            $asset = Asset::where('serial_no', $item['serialNumber'])->first();
            $assetId = $asset ? $asset->id : 0;
        }

        $data['user_id'] = $userId;
        $data['asset_id'] = $assetId;
        $data['checkin_date'] = $item['lastSyncDateTime'];
        $data['user_email'] = $item['emailAddress'];
        $data['user_name'] = $item['userDisplayName'];
        $data['name'] = $item['deviceName'];
        $data['serial_no'] = $item['serialNumber'];
        $data['manufacturer'] = $item['manufacturer'];
        $data['model'] = $item['model'];
        $data['os_version'] = $item['osVersion'];
        $data['os'] = $item['operatingSystem'];
        $data['mac_address'] = $item['wiFiMacAddress'] ? formatMacAddress($item['wiFiMacAddress']) : '';
        $data['ad_device_id'] = $item['azureADDeviceId'];
        $data['intune_id'] = $item['id'];
        $data['disk_capacity'] = $item['totalStorageSpaceInBytes'] / 1048576;
        $data['encrypted'] = $item['isEncrypted'];
        $data['activation_lock'] = !empty($item['activationLockBypassCode']) ? true : false;
        $data['platform'] = $item['operatingSystem'];

        return $data;
    }


    /**
     * Update the Intune Devices from the API Response
     */
    public function updateIntuneDevices($item)
    {
        $data = $this->getData($item);
        if (strlen($data['serial_no']) > 0 && $data['serial_no'] != "0") {
            $assetTypeId   = AssetType::firstOrCreate(['name' => 'Computer', 'slug' => 'computer'])->id;
            $assetStatusId = AssetStatus::firstOrCreate(['name' => 'Assigned', 'slug' => 'assigned'])->id;

            $intuneId = IntuneComputer::updateOrCreate(['serial_no' => $data['serial_no']], $data)->id;

            $data['asset_type_id'] = $assetTypeId;

            // Auto hardware update if asset exists
            if ($data['asset_id'] > 0 && $data['model']) {
                $techSpec = $this->getTechSpec($data, 'intune_model');
                $this->autoHardwareUpdate($data, $techSpec, "Intune");
            }


            //Auto create and assign asset if asset is not exist
            if (($data['asset_id'] == 0) && ($data['user_id'] != 0)) {
                $techSpec = $this->getTechSpec($data, 'intune_model');
                $data['asset_status_id'] = $assetStatusId;
                $data['intune_id']       = $intuneId;

                $this->autoCreateAndAssign($data, $techSpec, "Intune");
            }
        }
    }

    /** 
     * GET Techspec string for finding tech specs
     * @param mixed $intuneData
     * 
     * @return [type]
     */
    public function getTechSpecString($intuneData)
    {
        return convertSpecsToReadableValue($intuneData['disk_capacity'], 'disk_capacity');
    }

    /**
     *  Update Dicovery tool table asset ID
     * @param mixed $intuneData
     * @param mixed $assetId
     * 
     * @return [type]
     */
    public function updateDiscoveryAssetId($intuneData, $assetId)
    {
        return IntuneComputer::find($intuneData['intune_id'])->update(['asset_id' => $assetId]);
    }

    /**
     * Auto re assigns the discreapancy assets.
     * @return [type]
     */
    public function reassignAssets()
    {
        $intuneAssets = $this->repository->getUserMismatchDevices()->has('user')->get();
        $this->reAssignDiscrepancyAssets($intuneAssets, 'intune');
        return true;
    }
}
