<?php

namespace App\Services\DiscoveryTools\Intune;

abstract class AbstractIntuneService
{
    abstract public function data();
    abstract public function exportData();
    /**
     * Executing the query
     * @param $devices Query
     * @return Query Result
     */
    public function getOutputData($devices, $order)
    {
        $start = request('start');
        $limit = request('length');
        if ($limit != -1) {
            $devices = $devices->offset($start)->limit($limit);
        }

        $devices->orderBy($order, 'DESC');

        return $devices->get();
    }

    /**
     * Generating data table data from qurey result
     * @param $users Query Result
     * @param $start Starting index
     * @param $data Collection valriable that will return
     *
     * @return array
     */
    public function getReportData($assets, $start, $data = [])
    {
        $parentIndex = $start;

        foreach ($assets as $key => $asset) {
            $parentIndex++;
            $userLink = $asset->user_id ? route('users.show', $asset->user_id) : '#';
            $nestedData = $this->getNestedData($asset, $parentIndex, $userLink);
            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Taking the input data
     * @return Array
     */
    public function getInputData()
    {
        return [
            'serial_no'                     => isset(request('form')['serial_no']) ? request('form')['serial_no'] : '',
            'mac_address'                   => isset(request('form')['mac_address']) ? request('form')['mac_address'] : '',
            'model'                         => isset(request('form')['model']) ? request('form')['model'] : '',
            'enrolled_user'                 => isset(request('form')['enrolled_user']) ? request('form')['enrolled_user'] : '',
            'status'                        => isset(request('form')['user_status']) ? request('form')['user_status'] : '',
            'os_version'                    => isset(request('form')['os_version']) ? request('form')['os_version'] : '',
            'checkin_date_from'             => isset(request('form')['check_in_from']) ? request('form')['check_in_from'] : '',
            'checkin_date_to'               => isset(request('form')['check_in_to']) ? request('form')['check_in_to'] : '',
            'user_id'                       => isset(request('form')['user']) ? request('form')['user'] : '',
            'asset_type_id'                 => isset(request('form')['asset_type']) ? request('form')['asset_type'] : '',
            'manufacturer_id'               => isset(request('form')['manufacturer']) ? request('form')['manufacturer'] : '',
            'make_and_model_id'             => isset(request('form')['teq_model']) ? request('form')['teq_model'] : '',
            'location_id'                   => isset(request('form')['room_name']) ? request('form')['room_name'] : '',
            'asset_status_id'               => isset(request('form')['asset_status']) ? request('form')['asset_status'] : '',
            'platform'                      => isset(request('form')['platform']) ? request('form')['platform'] : '',
            'teqtivity_asset'               => isset(request('form')['teqtivity_asset']) ? request('form')['teqtivity_asset'] : '',
            'user_status_intune'            => isset(request('form')['user_status_intune']) ? request('form')['user_status_intune'] : '',
            'user_department_intune'        => isset(request('form')['user_department_intune']) ? request('form')['user_department_intune'] : '',
            'work_location_intune'          => isset(request('form')['work_location_intune']) ? request('form')['work_location_intune'] : '',
            'user_status'                   => isset(request('form')['user_status']) ? request('form')['user_status'] : '',
            'user_department'               => isset(request('form')['user_department']) ? request('form')['user_department'] : '',
            'work_location'                 => isset(request('form')['work_location']) ? request('form')['work_location'] : '',
            'host_name'                     => isset(request('form')['host_name']) ? request('form')['host_name'] : '',
            'imei'                          => isset(request('form')['imei']) ? request('form')['imei'] : '',
            'carrier'                       => isset(request('form')['carrier']) ? request('form')['carrier'] : '',
            'carrier_country'               => isset(request('form')['carrier_country']) ? request('form')['carrier_country'] : '',
        ];
    }

    /**
     * Taking the filters for Report generator Export
     *
     * @return Array
     */
    public function getExportInputdata()
    {
        return [
            'serial_no'                   => request('serial_no') ?? '',
            'mac_address'                 => request('mac_address') ?? '',
            'asset_type_id'               => request('asset_type') ?? '',
            'manufacturer_id'             => request('manufacturer') ?? '',
            'make_and_model_id'           => request('teq_model') ?? '',
            'model'                       => request('model') ?? '',
            'user_id'                     => request('user') ?? '',
            'enrolled_user'               => request('enrolled_user') ?? '',
            'status'                      => (request('user_status') !== null)  ? request('user_status') : '',
            'os_version'                  => request('os_version') ?? '',
            'checkin_date_from'           => request('check_in_from') ?? '',
            'checkin_date_to'             => request('check_in_to') ?? '',
            'asset_status_id'             => request('asset_status') ?? '',
            'location_id'                 => request('room_name') ?? '',
            'platform'                    => request('platform') ?? '',
            'teqtivity_asset'             => request('teqtivity_asset') ?? '',
            'user_status_intune'          => request('user_status_intune') ??  '',
            'work_location_intune'        => request('work_location_intune') ??  '',
            'user_department_intune'      => request('user_department_intune') ?? '',
            'user_status'                 => request('user_status') ??  '',
            'work_location'               => request('work_location') ??  '',
            'user_department'             => request('user_department') ?? '',
            'imei'                        => request('imei') ??  '',
            'host_name'                   => request('host_name') ??  '',
            'carrier'                     => request('carrier') ?? '',
            'carrier_country'             => request('carrier_country') ?? '',
        ];
    }

    public function getNestedData($asset, $index, $userLink)
    {

        $userLink   = $asset->user_id ? route('users.show', $asset->user_id) : 'javascript:void(0)';
        $assetLink  = $asset->asset ? route('assets.show', $asset->asset->id) : 'javascript:void(0)';
        $serialNo  = $asset->asset ? ("<a href=$assetLink>" . $asset->serial_no . "</a>") : $asset->serial_no;

        $intuneUser = $asset->user_id ? ("<a href=$userLink>" . $asset->user_email . "</a>") : $asset->user_email;

        $teqtivityUserLink   = isset($asset->asset) ? (optional($asset->asset)->user_id ? route('users.show', optional($asset->asset)->user_id) : 'javascript:void(0)') : '';
        $teqtivityUser = isset($asset->asset) ? ($asset->asset->user_id ? ("<a href=$teqtivityUserLink>" . optional(optional($asset->asset)->user)->first_name . ' - ' . optional(optional($asset->asset)->user)->email . "</a>") : '') : '';


        $nestedData['id']           = $index;
        $nestedData['serial_no']    = $serialNo;
        $nestedData['computer_name'] = $asset->name;
        $nestedData['asset_type'] = optional(optional($asset->asset)->assetType)->name;

        $nestedData['manufacturer_teqtivity']     = optional(optional(optional($asset->asset)->makeAndModel)->manufacturer)->name;
        $nestedData['hardware_teqtivity']            = optional(optional($asset->asset)->makeAndModel)->name;
        $nestedData['teqtivity_asset'] = $asset->asset_id ? 'Yes' : 'No';
        $nestedData['location_teqtivity']     = optional(optional($asset->asset)->location)->room_name;
        $nestedData['asset_status_teqtivity']     = $asset->asset ? optional($asset->asset->assetStatus)->name : '';


        $nestedData['intune_user']  = $intuneUser;
        $nestedData['intune_user_status']  = isset($asset->user) ? (optional($asset->user)->status == 1 ? 'Active' : 'Terminated') : '';
        $nestedData['intune_user_department']  = optional(optional($asset->user)->department)->name;
        $nestedData['intune_user_work_location']  = optional($asset->user)->city;


        $nestedData['teqtivity_user']   = $teqtivityUser;
        $nestedData['teqtivity_user_status']  = isset($asset->asset->user) ? (optional(optional($asset->asset)->user)->status == 1 ? 'Active' : 'Terminated') : '';
        $nestedData['teqtivity_user_department']  = optional(optional(optional($asset->asset)->user)->department)->name;
        $nestedData['teqtivity_user_work_location']  = optional(optional($asset->asset)->user)->city;


        $nestedData['checkin_date'] = $asset->checkin_date;
        $nestedData['os_version']   = $asset->os_version;
        $nestedData['hardware']   = $asset->model;
        $nestedData['mac_address']   = $asset->mac_address;

        $nestedData['last_modified']    = parse_date_from_db_datetime($asset->last_modified_date);

        $nestedData['imei']             = $asset->imei;
        $nestedData['last_modified_date']   = (($asset->asset) && ($asset->asset->latestAssetHistory)) ?  $asset->asset->latestAssetHistory->updated_at : (($asset->latestAssetHistory) ?  $asset->latestAssetHistory->updated_at : '');
        $nestedData['platform']    = $asset->platform;
        $nestedData['update_asset']     = $this->getUpdateDiscoveryAssetButton($asset->asset, optional($asset->asset)->assetStatus);
        $nestedData['create_asset']     = $this->getCreateAssetButton($asset);




        return $nestedData;
    }

    /**
     * Creating array for export data
     * @param $asset Query Result
     * @param $requiredColumns columns to be added in export file
     *
     * @return array
     */
    public function getExportNestedData($asset, $requiredColumns)
    {
        $nestedData['Serial # (Intune)']                 = $asset->serial_no;
        $nestedData['Asset Type (Teqtivity)'] = optional(optional($asset->asset)->assetType)->name;
        $nestedData['Computer Name']            = $asset->name;
        $nestedData['OS Platform']               = $asset->platform;
        $nestedData['OS Version']               = $asset->os_version;
        $nestedData['Last Check In']            = $asset->checkin_date;
        $nestedData['Last Modified Date']       = (($asset->asset) && ($asset->asset->latestAssetHistory)) ?  $asset->asset->latestAssetHistory->updated_at : (($asset->latestAssetHistory) ?  $asset->latestAssetHistory->updated_at : '');

        $nestedData['Manufacturer (Teqtivity)'] = optional(optional(optional($asset->asset)->makeAndModel)->manufacturer)->name;
        $nestedData['Hardware Standard (Teqtivity)'] = optional(optional($asset->asset)->makeAndModel)->name;
        $nestedData['Asset Location (Teqtivity)'] = optional(optional($asset->asset)->location)->room_name;
        $nestedData['Asset Status (Teqtivity)'] = $asset->asset ? optional($asset->asset->AssetStatus)->name : '';

        $nestedData['User (Intune)'] = optional($asset->user)->email;
        $nestedData['User Status (Intune)'] = isset($asset->user) ? (optional($asset->user)->status == 1 ? 'Active' : 'Terminated') : '';
        $nestedData['User Department (Intune)'] = optional(optional($asset->user)->department)->name;
        $nestedData['User Work Location (Intune)'] = optional($asset->user)->city;

        $nestedData['User (Teqtivity)'] = optional(optional($asset->asset)->user)->first_name . ' - ' . optional(optional($asset->asset)->user)->email;
        $nestedData['User Status (Teqtivity)'] = isset($asset->asset->user) ? (optional(optional($asset->asset)->user)->status == 1 ? 'Active' : 'Terminated') : '';
        $nestedData['User Department (Teqtivity)'] =  optional(optional(optional($asset->asset)->user)->department)->name;
        $nestedData['User Work Location (Teqtivity)'] =  optional(optional($asset->asset)->user)->city;

        $nestedData['Hardware Standard (Intune)'] = $asset->model;
        $nestedData['MAC Address (Intune)'] = $asset->mac_address;
        $nestedData['Last Seen'] = $asset->checkin_date;


        foreach ($requiredColumns as $key => $column) {
            $data[$column] =  $nestedData[$column];
        }
        return $data;
    }

    /**
     * Filer the results based on filters
     *
     * @param $assets DB Query
     * @param Array $inputData   Filter data
     *
     * @return Query
     */
    public function filterDevices($assets, $inputData)
    {
        $assets     = $this->intuneRepository->filterRelation($assets, 'serial_no', $inputData['serial_no']);
        $assets     = $this->intuneRepository->filterRelation($assets, 'model', $inputData['model']);
        $assets     = $this->intuneRepository->filterRelation($assets, 'os_version', $inputData['os_version']);
        $assets     = $this->intuneRepository->filterRelation($assets, 'user_email', $inputData['enrolled_user']);
        $assets     = $this->intuneRepository->filterRelation($assets, 'platform', $inputData['platform']);
        $assets     = $this->intuneRepository->filterRelation($assets, 'mac_address', $inputData['mac_address']);

        $assets     = $this->intuneRepository->filterWithAssetMakeAndModelFields($assets, 'manufacturer_id', $inputData['manufacturer_id']);

        $assets     = $this->intuneRepository->filterWithAssetFields($assets, 'user_id', $inputData['user_id']);
        $assets     = $this->intuneRepository->filterWithAssetFields($assets, 'asset_type_id', $inputData['asset_type_id']);
        $assets     = $this->intuneRepository->filterWithAssetFields($assets, 'make_and_model_id', $inputData['make_and_model_id']);
        $assets     = $this->intuneRepository->filterWithAssetFields($assets, 'location_id', $inputData['location_id']);
        $assets     = $this->intuneRepository->filterWithAssetFields($assets, 'asset_status_id', $inputData['asset_status_id']);

        $assets     = $this->intuneRepository->filterWithUserFields($assets, 'status', $inputData['status']);

        $assets     = $this->intuneRepository->filterWithUserFields($assets, 'status', $inputData['user_status_intune']);
        $assets     = $this->intuneRepository->filterWithUserFields($assets, 'city', $inputData['work_location_intune']);
        $assets     = $this->intuneRepository->filterWithUserFields($assets, 'department_id', $inputData['user_department_intune']);

        $assets     = $this->intuneRepository->filterWithTeqtivityUserFields($assets, 'department_id', $inputData['user_department']);
        $assets     = $this->intuneRepository->filterWithTeqtivityUserFields($assets, 'status', $inputData['user_status']);
        $assets     = $this->intuneRepository->filterWithTeqtivityUserFields($assets, 'city', $inputData['work_location']);

        $assets     = $this->intuneRepository->filterCheckInDateFrom($assets, $inputData['checkin_date_from']);
        $assets     = $this->intuneRepository->filterCheckInDateTo($assets, $inputData['checkin_date_to']);

        $assets     = $this->intuneRepository->filterTeqtivityAssetOrNot($assets, $inputData['teqtivity_asset']);

        return $assets;
    }

    /**
     * Generating data to export
     * @param $assets Query Result
     * @param $start Starting index
     * @param $data Collection valriable that will return
     * @param $requiredColumns colums to be shown in export file

     * @return array
     */
    public function getReportExportData($assets, $start, $data, $requiredColumns)
    {
        $parentIndex = $start;

        foreach ($assets as $key => $asset) {
            $parentIndex++;
            $nestedData = $this->getExportNestedData($asset, $requiredColumns);
            $data[] = $nestedData;
        }

        return $data;
    }

    /**
     * Retrieves the HTML code for the update discovery asset button.
     *
     * @param mixed $asset The asset object.
     * @param mixed $assetStatus The asset status object.
     * @return string Returns the HTML code for the update discovery asset button.
     */
    public function getUpdateDiscoveryAssetButton($asset, $assetStatus)
    {
        if (auth()->user()->can('IT Assets Edit') && $asset) {
            return '<a href="javascript:;" data-asset="' . $asset->id . '" data-asset-serial="' . $asset->serial_no . '" data-asset-current-status="' . optional($assetStatus)->slug . '" data-parent-asset="' . $asset->parent_asset_id . '" class="btn btn-primary btn-sm update-discrepancy-asset">Update Asset</a>';
        }
        return '';
    }

    /**
     * Set up Create Asset Button
     * @param mixed $asset
     */
    public function getCreateAssetButton($asset)
    {
        if (auth()->user()->can('IT Assets Edit') && $asset) {
            return '<a href="javascript:;" data-asset="' . $asset->id . '" data-asset-serial="' . $asset->serial_no . '" class="btn btn-primary btn-sm create-asset">Create Asset</a>';
        }
        return '';
    }
}
