<?php

namespace App\Services\DiscoveryTools\Chromebook;

use App\Http\Traits\AutoCreateAssignTrait;
use App\Models\Asset;
use App\Models\AssetStatus;
use App\Models\AssetType;
use App\Models\Chromebook;
use App\Repositories\DiscoveryTools\ChromebookRepository;
use App\Services\Integrations\GoogleWorkspace\GoogleWorkspaceIntegration;
use App\User;
use Cache;
use Exception;

class ChromeBookSyncService
{
    use AutoCreateAssignTrait;

    public function __construct(protected GoogleWorkspaceIntegration $integration, protected ChromebookRepository $repository) {}

    /**
     * Retrieves the JWT access token for the Google Directory API
     *
     * @return string|false The JWT access token or false if an error occurred
     */
    public function getJwtAccessToken()
    {
        return $this->integration->getAccessTokenWithJwt('chromebook');
    }

    /**
     * Imports the Chrome OS devices from the Google Workspace API
     *
     * @return int The number of Chrome OS devices imported
     */
    public function importChromeOsdevices()
    {
        $token = $this->getJwtAccessToken();

        $pageToken = '';
        $count = 0;

        while (1) {

            $response = $this->integration->getChromeOsdevices($token, $pageToken);

            if ($response === false) {
                return null;
            }

            if (!empty($response['nextPageToken'])) {
                $pageToken = $response['nextPageToken'];
            }

            if (isset($response['chromeosdevices'])) {
                $count += $this->saveChromeDevices($response['chromeosdevices']);
            }

            if ($pageToken == '') {
                break;
            }
        }

        return $count;
    }

    /**
     * Save API data to the database.
     *
     * @param mixed $chromeOsDevices The API data to be saved.
     *
     * @return int The count of successfully saved records.
     */
    public function saveChromeDevices($chromeOsDevices)
    {
        $count = 0;

        foreach ($chromeOsDevices as $chromeosdevice) {
            $hardwareData = [];

            try {
                $this->updateChromeBookData($chromeosdevice, $hardwareData);
                $count++;
            } catch (Exception $e) {
                // dd($e->getMessage(), $e->getLine(), $e->getFile());
                continue;
            }
        }

        return $count;
    }

    /**
     * Update the ChromeBook records based on the API response
     * 
     * @param $item API response
     */
    public function updateChromeBookData($item, $hardwareData)
    {
        $data = $this->getCronData($item);
        if (strlen($data['serial_no']) > 0) {
            $assetTypeId   = AssetType::firstOrCreate(['name' => 'Computer', 'slug' => 'computer'])->id;
            $assetStatusId = AssetStatus::firstOrCreate(['name' => 'Assigned', 'slug' => 'assigned'])->id;
            $chromebookId  = Chromebook::updateOrCreate(['serial_no' => $data['serial_no']], $data)->id;

            // $data['make']          = optional($hardwareData->hardware)->make;
            $data['poNumber']      = NULL;
            $data['asset_tag']     = NULL;
            $data['asset_type_id'] = $assetTypeId;

            //Auto Create and Assign Asset
            if (($data['asset_id'] == 0) && ($data['user_id'] != 0)) {
                $techSpec = $this->getTechSpec($data, 'chrome_model');
                $data['asset_status_id'] = $assetStatusId;
                $data['chromebook_id']   = $chromebookId;
                $this->autoCreateAndAssign($data, $techSpec, "Chromebook");
            }
        }
    }

    /**
     * Making array with needed data for saving to DB
     * @param $item CSV data array
     * @return array
     */
    public function getCronData($item)
    {
        $email                        = $this->getEmailFromApiResponse($item['recentUsers'] ?? []);
        $userId                       = $this->getUserId($email);

        $lastSync                     = $this->getChromeBookTime($item['lastSync'] ?? '');
        $lastEnrollmentTime           = $this->getChromeBookTime($item['lastEnrollmentTime'] ?? '');

        $data['user_id']              = $userId;
        $data['model']                = $item['model'] ?? '';
        $data['serial_no']            = $this->getSerialNumber($data['model'], $item['serialNumber']);

        $asset                        = $this->getAsset($data['serial_no']);

        $data['asset_id']             = $asset ? $asset->id : 0;
        $data['mac_address']          = $item['mac_address'] ?? '';
        $data['status']               = $item['status'] == 'ACTIVE' ? 1 : 0;
        $data['chrome_status']        = $item['status'] ?? '';
        $data['email']                = $email; //$item['annotatedUser'] ?? '';
        $data['last_sync']            = $lastSync;
        $data['last_enrollment_time'] = $lastEnrollmentTime;
        $data['platform']             = $item['platformVersion'] ?? '';
        $data['os_version']           = $item['osVersion'] ?? '';
        $data['manufacturer']         = isset($item['model']) ? strval(explode(' ', $item['model'])[0]) : null;
        $data['mac_address']          = $item['macAddress'] ?? '';
        $data['ram']                  = $this->getDeviceRamInfo($item);
        $data['disk_capacity']        = $this->getDiskVolumeInfo($item);
        $data['processor']            = $this->getProcessorFromCpuInfo($item);
        $data['processor_speed']      = $this->getProcessorSpeed($item);
        $data['device_id']            = $item['deviceId'] ?? '';


        //update the Asset Attributes
        $this->updateAssetDetails($asset, $data);


        return $data;
    }

    /**
     * Taing the email from Recent Users index
     */
    public function getEmailFromApiResponse($recentUsers)
    {
        if (empty($recentUsers)) {
            return null;
        }

        return $recentUsers[0] ? ($recentUsers[0]['email'] ?? null) : null;
    }

    public function getUserId($email = '')
    {
        if ($email) {
            $user = User::where('email', $email)->first();

            return  $user ? $user->id : 0;
        }

        return 0;
    }

    /**
     * get the Asset id from serial #
     *
     * @param string $serial
     * @return mixed
     */
    public function getAsset($serial)
    {
        if ($serial) {
            $asset = Asset::where('serial_no', $serial)->first();

            //return $asset ? $asset->id : 0;
            return $asset;
        }

        return 0;
    }

    public function getChromeBookTime($timeString = '')
    {
        if ($timeString) {
            $date = strtotime($timeString);

            return date('Y-m-d h:i:s A', $date);
        }

        return null;
    }

    /**
     * update Asset Details
     *
     * @param Asset $assets
     * @param Array $data
     *
     * @return bool
     */
    public function updateAssetDetails($asset, $data)
    {
        if ($asset) {
            if (empty($asset->mac) && !empty($data['mac_address'])) {
                $asset->mac = $data['mac_address'];
                $asset->update();
            }
        }

        return true;
    }

    /**
     * function to limit last 8 characters of serialNumber when model is `Lenovo ThinkPad C13`
     * 
     * @param string $model
     * @param string $serialNumber
     * 
     * @return string
     */
    private function getSerialNumber($model, $serialNumber)
    {
        if (strpos($model, 'Lenovo ThinkPad C13') !== false) {
            return substr($serialNumber, -8);
        }

        return $serialNumber ?? '';
    }

    /** 
     * GET Techspec string for finding tech specs
     * @param mixed $chromeData
     * 
     * @return [type]
     */
    public function getTechSpecString($chromeData)
    {
        //return convertKilohertzToGigahertz($chromeData['processor_speed']) . "Ghz / " . convertSpecsToReadableValueFromBytes($chromeData['ram'], 'ram') . " / " . $chromeData['disk_capacity'] . 'GB';
        $processorSpeed = convertSpecsToReadableValue($chromeData['processor_speed'], 'processor_speed');
        return ($processorSpeed == '-' ? 'Ghz' : $processorSpeed) . " / " . convertSpecsToReadableValue($chromeData['ram'], 'ram') . " / " . convertSpecsToReadableValue($chromeData['disk_capacity'], 'disk_capacity');
    }

    /**
     * Update Dicovery tool table asset ID
     * @param mixed $chromeData
     * @param mixed $assetId
     * 
     * @return [type] 
     */
    public function updateDiscoveryAssetId($chromeData, $assetId)
    {
        Chromebook::find($chromeData['chromebook_id'])->update(['asset_id' => $assetId]);
    }

    /**
     * Auto re assigns the discreapancy assets.
     * @return [type]
     */
    public function reassignAssets()
    {
        $discrepancyDevices = $this->repository->getUserMismatchDevices()->has('user')->get();
        $this->reAssignDiscrepancyAssets($discrepancyDevices, 'Chromebook');
        return true;
    }

    /**
     * Gets the cpu model
     * @param mixed $item
     * 
     * @return [type]
     */
    private function getProcessorFromCpuInfo($item)
    {
        $processor = NULL;
        if (isset($item['cpuInfo'])) {
            $processor = $item['cpuInfo'][0]['model'] ?? NULL;
        }
        return $processor;
    }
    /**
     * Finds the device processor speed (Khz) , converts to Mhz before return
     * @param mixed $item
     * 
     * @return [type]
     */
    private function getProcessorSpeed($item)
    {
        $processorSpeed = NULL;
        if (isset($item['cpuInfo'])) {
            $processorSpeed = isset($item['cpuInfo'][0]['maxClockSpeedKhz']) ? ($item['cpuInfo'][0]['maxClockSpeedKhz'] / 1024) : NULL;
        }
        return $processorSpeed;
    }
    /**
     * Finds the device ram , which comes in kb returns value converted to MB 
     * @param mixed $item
     * 
     * @return [type]
     */
    private function getDeviceRamInfo($item)
    {
        $ram = NULL;
        if (isset($item['systemRamTotal'])) {
            $ram = $item['systemRamTotal'] / 1024 / 1024;
        }
        return $ram;
    }
    /**
     * Finds the device volume Information coverts to MB before return
     * @param mixed $item
     * 
     * @return [type]
     */
    private function getDiskVolumeInfo($item)
    {
        $diskCapacity = 0;
        $diskCapacity = $item['diskVolumeReports'][0]['volumeInfo'][0]['storageTotal'] ?? 0;
        return ($diskCapacity / 1024 / 1024);
    }
}
