<?php

namespace App\Services\Dashboard;

use App\Repositories\UserRepository;
use App\Repositories\TermAssets;

class HrDashboardService
{

    public function __construct(protected UserRepository $userRepository, protected TermAssets $termAssetsRepo,)
    {
    }

    /**
     * It gets the number of new hired users in full time and external  and the number of assets they have, and returns a table of data. 
     * 
     * @param fromDate The start date of the date range you want to query.
     * @param toDate The date to end the report on.
     * @return array of summary data
     */
    public function getOnboardingSummary($fromDate, $toDate)
    {
        $fulltimeNewhireUsers = $this->userRepository->getNewHires()->countDesktopOrLaptopAssigned();
        $fulltimeNewhireUsers = $this->userRepository->searchNewhireUserWithFromDate($fulltimeNewhireUsers, $fromDate);
        $fulltimeNewhireUsers = $this->userRepository->searchUserWithToDate($fulltimeNewhireUsers, $toDate);

        $fulltimeUsers = $this->userRepository->getFteUsers($fulltimeNewhireUsers);
        $fulltimeUserCount = $fulltimeUsers->count();

        $extNewhireUsers = $this->userRepository->getNewHires()->countDesktopOrLaptopAssigned();
        $extNewhireUsers = $this->userRepository->searchNewhireUserWithFromDate($extNewhireUsers, $fromDate);
        $extNewhireUsers = $this->userRepository->searchUserWithToDate($extNewhireUsers, $toDate);
        $extUsers = $this->userRepository->getExtUsers($extNewhireUsers);
        $extUserCount = $extUsers->count();

        $fulltimeUserTotalAssets = array_sum(array_column($fulltimeUsers->get()->toArray(), 'assets_count'));
        $extUserTotalAssets = array_sum(array_column($extUsers->get()->toArray(), 'assets_count'));
        return $this->getSummaryTableData($fulltimeUserCount, $fulltimeUserTotalAssets, $extUserCount, $extUserTotalAssets, '/new-hire', $fromDate, $toDate);
    }


    /**
     * It gets the count of terminated fulltime users and the sum of their assets, and the count of
     * terminated external users and the sum of their assets
     * 
     * @param fromDate The start date of the period you want to get the summary for.
     * @param toDate The date to end the report on.
     */
    public function getTerminationSummary($fromDate, $toDate)
    {
        $fulltimeUsers     = $this->termAssetsRepo->getTermAssets();
        $fulltimeUsers     = $this->termAssetsRepo->searchUserWithFromDate($fulltimeUsers, $fromDate);
        $fulltimeUsers     = $this->termAssetsRepo->searchUserWithToDate($fulltimeUsers, $toDate);
        $fulltimeUsers     = $this->userRepository->getFteUsers($fulltimeUsers);
        $fulltimeUserCount = $fulltimeUsers->count();

        $extUsers          = $this->termAssetsRepo->getTermAssets();
        $extUsers          = $this->termAssetsRepo->searchUserWithFromDate($extUsers, $fromDate);
        $extUsers          = $this->termAssetsRepo->searchUserWithToDate($extUsers, $toDate);
        $extUsers          = $this->userRepository->getExtUsers($extUsers);
        $extUserCount      = $extUsers->count();

        $fulltimeUserAssetsCount =  array_sum(array_column($fulltimeUsers->get()->toArray(), 'term_asset_count'));
        $extUserAssetsCount = array_sum(array_column($extUsers->get()->toArray(), 'term_asset_count'));
        return $this->getSummaryTableData($fulltimeUserCount, $fulltimeUserAssetsCount, $extUserCount, $extUserAssetsCount, '/terms-report', $fromDate, $toDate);
    }

    /**
     * Retrieves the summary table data for the given parameters.
     *
     * @param int $fulltimeUserCount The count of full-time users.
     * @param int $fulltimeUserAssetsSum The sum of assets for full-time users.
     * @param int $extUserCount The count of external users.
     * @param int $extUserAssetsSum The sum of assets for external users.
     * @param string $link The link to be appended to the URL.
     * @param mixed $fromDate The starting date for the data retrieval.
     * @param mixed $toDate The ending date for the data retrieval.
     * @return array The summary table data.
     */
    public function getSummaryTableData($fulltimeUserCount, $fulltimeUserAssetsSum, $extUserCount, $extUserAssetsSum, $link = '', $fromDate = "", $toDate = "")
    {
        $url = url($link);
        $data[0]['user_type'] = 'Full Time';
        $data[0]['user_count'] = '<a  target="_blank" href="' . $url . '?fte=full_time&from=' . urlencode($fromDate) . '&to=' . urlencode($toDate) . '">' . $fulltimeUserCount . '</a>';
        $data[0]['asset_count'] = $fulltimeUserAssetsSum;
        $data[1]['user_type'] = 'Ext';
        $data[1]['user_count'] = '<a  target="_blank" href="' . $url . '?fte=ext&from=' . urlencode($fromDate) . '&to=' . urlencode($toDate) . '">' . $extUserCount . '</a>';
        $data[1]['asset_count'] = $extUserAssetsSum;
        return $data;
    }
}
