<?php

namespace App\Services\Dashboard;

use App\Models\AssetStatus;
use App\Models\AirWatch;
use App\Models\IntuneComputer;
use App\Models\Chromebook;
use App\Models\EolWipedCheckingInAssetsCount;
use App\Models\JamfComputer;
use App\Models\KandjiDevices;
use App\Models\MobileIron;
use Illuminate\Support\Facades\DB;

class EOLWipedCheckingInAssetsCountService
{

    /**
     * It takes the count of assets for the given types and stores it in the database
     *
     * @param months The number of months you want to go back in time.
     */
    public function updateEOLWipedCheckingInAssetsCount()
    {
        $types = ["laptop" => 'Laptops', "desktop" => 'Desktops', 'chromebooks' => 'Chromebooks', "tablet" => 'Tablets', "mobile_phone" => 'Mobiles'];
        foreach ($types as $type => $typo) {
            $count = $this->getEOLWipedCheckingInData($type);
            EOLWipedCheckingInAssetsCount::updateOrCreate(
                ['type' => $typo],
                ['count' => $count]
            );
        }
    }

    /**
     * to get the count of  devices that are EOL Wiped status but they are checking in
     *
     */
    function  getEOLWipedCheckingInData($type)
    {
        if ($type != 'chromebooks') {
            return $this->getEOLWipedCheckingInDataForAssettype($type);
        }
        return $this->getEOLWipedCheckingInDataForChromebooks();
    }

    /**
     * Get the EOLWiped Checking In Data For Assettypes
     * @param mixed $type
     * 
     * @return int
     */
    public function getEOLWipedCheckingInDataForAssettype($type)
    {
        $statusId = AssetStatus::getStatusID('end_of_life_data_wiped')->get()->pluck('id');

        $workspaceAssets = $this->getAssetCount($type, AirWatch::class, $statusId, 'air_watches', 'last_seen');

        $chromebookAssets = $this->getAssetCount($type, Chromebook::class, $statusId, 'chromebooks', 'last_sync');

        $intuneAssets = $this->getAssetCount($type, IntuneComputer::class, $statusId, 'intune_computers', 'checkin_date');

        $jamfAssets = $this->getAssetCount($type, JamfComputer::class, $statusId, 'jamf_computers', 'checkin_date');

        $kandjiAssets = $this->getAssetCount($type, KandjiDevices::class, $statusId, 'kandji_devices', 'last_checkin');

        $mobileIronAssets = $this->getAssetCount($type, MobileIron::class, $statusId, 'mobile_irons', 'last_checkin');

        $totalAssetsCount = $workspaceAssets + $chromebookAssets + $intuneAssets + $jamfAssets + $kandjiAssets + $mobileIronAssets;

        return $totalAssetsCount;
    }

    /**
     * Making query to get the count of  devices that are EOL Wiped status but they are checking in
     * for assettypes such as laptop, desktop, mobile and tablets
     * @param mixed $type
     * @param mixed $model
     * @param mixed $statusId
     * @param mixed $joinTable
     * @param mixed $lastSeenColumn
     * 
     * @return int
     */
    private function getAssetCount($type, $model, $statusId, $joinTable, $lastSeenColumn)
    {
        return $model::whereHas('asset', function ($query) use ($statusId) {
            $query->whereIn('asset_status_id', $statusId);
        })
            ->whereHas('asset.assetType', function ($query) use ($type) {
                $query->where('slug', $type);
            })
            ->leftJoin("asset_histories", function ($join) use ($statusId, $joinTable) {
                $join->on('asset_histories.asset_id', '=', $joinTable . '.asset_id')
                    ->on('asset_histories.id', '=', DB::raw("(SELECT MAX(id) from asset_histories where asset_histories.asset_id = " . $joinTable . ".asset_id)"))
                    ->where('asset_histories.new_asset_status_id', '=', $statusId);
            })
            ->where(function ($query) use ($joinTable, $lastSeenColumn) {
                $query->where($joinTable . '.' . $lastSeenColumn, '>', DB::raw('asset_histories.updated_at'))
                    ->orWhereNull('asset_histories.updated_at');
            })->count();
    }

    /**
     * Get the EOLWiped Checking In Data For Chromebook
     * @return int
     */
    public function getEOLWipedCheckingInDataForChromebooks()
    {
        $statusId = AssetStatus::getStatusID('end_of_life_data_wiped')->get()->pluck('id');

        $workspaceAssets = $this->getChromebookAssetCount(AirWatch::class, $statusId, 'air_watches', 'last_seen');

        $chromebookAssets = $this->getChromebookAssetCount(Chromebook::class, $statusId, 'chromebooks', 'last_sync');

        $intuneAssets = $this->getChromebookAssetCount(IntuneComputer::class, $statusId, 'intune_computers', 'checkin_date');

        $jamfAssets = $this->getChromebookAssetCount(JamfComputer::class, $statusId, 'jamf_computers', 'checkin_date');

        $kandjiAssets = $this->getChromebookAssetCount(KandjiDevices::class, $statusId, 'kandji_devices', 'last_checkin');

        $mobileIronAssets = $this->getChromebookAssetCount(MobileIron::class, $statusId, 'mobile_irons', 'last_checkin');

        $totalAssetsCount = $workspaceAssets + $chromebookAssets + $intuneAssets + $jamfAssets + $kandjiAssets + $mobileIronAssets;

        return $totalAssetsCount;
    }

    /**
     * Making query to get the count of  devices that are EOL Wiped status but they are checking in for chromebooks
     * @param mixed $model
     * @param mixed $statusId
     * @param mixed $joinTable
     * @param mixed $lastSeenColumn
     * 
     * @return int
     */
    private function getChromebookAssetCount($model, $statusId, $joinTable, $lastSeenColumn)
    {
        return $model::whereHas('asset', function ($query) use ($statusId) {
            $query->whereIn('asset_status_id', $statusId);
        })
            ->whereHas('asset.makeAndModel', function ($query) {
                $query->where('slug', 'like', '%chromebook%');
            })
            ->leftJoin("asset_histories", function ($join) use ($statusId, $joinTable) {
                $join->on('asset_histories.asset_id', '=', $joinTable . '.asset_id')
                    ->on('asset_histories.id', '=', DB::raw("(SELECT MAX(id) from asset_histories where asset_histories.asset_id = " . $joinTable . ".asset_id)"))
                    ->where('asset_histories.new_asset_status_id', '=', $statusId);
            })
            ->where(function ($query)  use ($joinTable, $lastSeenColumn) {
                $query->where($joinTable . '.' . $lastSeenColumn, '>', DB::raw('asset_histories.updated_at'))
                    ->orWhereNull('asset_histories.updated_at');
            })->count();
    }
}
